/*

HBO144q16, non-stiff differential equations solver of order 14.

HBO144q16(double RELTOL, double TOL, double t0, double tend, double*y0,double*tvals,
 double**yvals, double**fvals, double**f2vals, double**f3vals, int*res,int nbcmp)	
 integrates the system of differential equations y' = f(t,y) from time t0 to tend with
   initial conditions y0 whose number of component is "nbcmp".

The formula of HBO144q16 uses y',  y'' and y''' as in Obrechkoff methods.
Simple formulae are derived once for all to obtain the values of Hermite-Birkhoff
 interpolation polynomials in terms of Lagrange basis functions
 for 16 quantized stepsize ratios. The coefficients of 
 these formulae are stored in "HBO144cfs.cpp"

 Each row in the solution array "yvals', derivatives "fvals","f2vals" and "f3vals"
  corresponds to a time returned in the column vector "tvals"
 
Input parameters to HBO144q16 --

  RELTOL -- not used in this code
  TOL -- scalar error tolerances
  t0 -- starting integration time
  tend -- final integration time
  y0 -- vector of initial conditions 
  nbcmp -- number of equations to be integrated

Output parameters from HBO144q16 --

  tvals -- independent variable
  yvals -- solution array
  fvals -- first derivative of solution array "yvals'
  f2vals -- second derivative of solution array "yvals'
  f3vals -- third derivative of solution array "yvals'
  res
	res[1] -- number of function evaluations ;
	res[2] -- number of success steps   ;
	res[3] -- number of failed attempts  ;

This code is explained in more details in
Hermite-Birkhoff-Obrechkoff 4-stage 4-step ODE solver of order 14
with quantized stepsize
Truong Nguyen-Ba, Philip W. Sharp, and Remi Vaillancourt
Journal C.A.M., submitted.


 */

#include <math.h>
#include <iostream>

#include includeName

using namespace std;


//  HBO7d123varopstep.m (6-stage)
static void HBO76_3dst(double RELTOL, double TOL, double t,double*y,double h,int j,double* tvals, double**yvals, double**fvals, double**f2vals, double**f3vals, int nbpos, int nbcmp,double*resHNEWandERK){
	int nbcmpP1=nbcmp+1;
	//
	// Post: tnew=tc+h, ynew is an approximate solution at t=tnew, and
	//	fnew = f(tnew, ynew)
	
	// Initialize method parameters.

    double h2,h3, erki ;	
	double tc=t;
	double stfac=0.81 ;
	
	double powD = 1.0/6.0;

// HBO7
// Ebv1_6 = [ 3062780599/29816473381-581429233/7480971313 ;  -1533181/7306695+9222278/50999049  ; 
//     605283937/1107746875-4539313/7992920 ; 41554493/75355849-3692668/11716025  ; 0 ;
//     720961/77876101-14791056/67227925 ] ; 
// Ebv7   = [-27007585/1439293967-1563392/45805487 ] ;
// Ebv8   = [ -67337485/46017752461-1015453/304980503 ];

	double* Ebv1_6 = new double[7];
	Ebv1_6[1]= 3062780599.0/29816473381.0 -581429233.0/7480971313.0     ;
	Ebv1_6[2]= -1533181.0/7306695.0 +9222278.0/50999049.0   ;
	Ebv1_6[3]= 605283937.0/1107746875.0 -4539313.0/7992920.0               ;
	Ebv1_6[4]= 41554493.0/75355849.0 -3692668.0/11716025.0         ;
	Ebv1_6[5]= 0 ;
	Ebv1_6[6]= 720961.0/77876101.0-14791056.0/67227925.0  ;		

	double Ebv7   = -27007585.0/1439293967.0-1563392.0/45805487.0    ;
	double Ebv8   = -67337485.0/46017752461.0-1015453.0/304980503.0 ;
	
	double** BMAT=new double*[7 ];
	for(int i=0;i<=6;i++){
		BMAT[i]=new double[7 ];
//		for(int jj=1;jj<=3;jj++)
//			BMAT[i][jj]=0.0;
	}


	
//% *** HBO7 B *****************
//% HBO7
//% c2    c3       c4             c5                    c6                    IF
//B=[
//   0     0        0              0                     0                3062780599/29816473381
//   0     0        0              0              -9749783/42956078       -1533181/7306695
//   0     0        0         -7617722/20101141   3327355531/11484129947   605283937/1107746875 
//   0     0     16384/2025    57594571/1807044    86767468/3359647        41554493/75355849 
//   0    81/320  -1088/75     -26032617/407999     -34264951556/628358387            0
//  1/5   3/64   14612/2025   150112116/4521287    83898588/2830417        720961/77876101  ];
//bv7= [ 1/50  ;  -9/1600 ;  536/675 ;  14035033/3600870 ;  28792462/7935665 ;   -27007585/1439293967 ];
//bv8= [ 1/750 ;  -9/16000 ; 64/5625 ;  4635333/57807374 ;  11257698/122298911 ; -67337485/46017752461    ];

	
	BMAT[1][6]=3062780599.0/29816473381.0  ;
	
	BMAT[2][5]= -9749783.0/42956078.0   ;
	BMAT[2][6]=  -1533181.0/7306695.0   ;
	
	BMAT[3][4]= -7617722.0/20101141.0 ;
	BMAT[3][5]=  3327355531.0/11484129947.0   ;
	BMAT[3][6]=  605283937.0/1107746875.0   ;

	BMAT[4][3]=  16384.0/2025.0     ;
	BMAT[4][4]=  57594571.0/1807044.0    ;
	BMAT[4][5]=  86767468.0/3359647.0   ;
	BMAT[4][6]=  41554493.0/75355849.0    ;

	BMAT[5][2]=  81.0/320.0     ;
	BMAT[5][3]=  -1088.0/75.0      ;
	BMAT[5][4]=  -26032617.0/407999.0   ;
	BMAT[5][5]=   -34264951556.0/628358387.0      ;
	BMAT[5][6]=  0.0  ;	

//  1/5   3/64   14612/2025   150112116/4521287    83898588/2830417        720961/77876101
	BMAT[6][1]=  1.0/5.0  ;
	BMAT[6][2]=  3.0/64.0   ;
	BMAT[6][3]=  14612.0/2025.0     ;
	BMAT[6][4]=  150112116.0/4521287.0    ;
	BMAT[6][5]=  83898588.0/2830417.0    ;
	BMAT[6][6]=  720961.0/77876101.0  ;	

//%       c2     c3         c4          c5                   c6                  IF
// HBO7 d123
//bv7= [ 1/50  ;  -9/1600 ;  536/675 ;  14035033/3600870 ;  28792462/7935665 ;   -27007585/1439293967 ];

	double* h2bv7=new double[7 ];
	h2=h*h ;
	h2bv7[0]=0.0;
	h2bv7[1]=h2*1.0/50.0 ;
	h2bv7[2]=h2* (-9.0/1600.0)  ;
	h2bv7[3]=h2* 536.0/675.0  ;
	h2bv7[4]=h2*  14035033.0/3600870.0  ;	
	h2bv7[5]=h2* 28792462.0/7935665.0   ;	
	h2bv7[6]=h2*  (-27007585.0/1439293967.0)  ;	

//bv8= [ 1/750 ;  -9/16000 ; 64/5625 ;  4635333/57807374 ;  11257698/122298911 ; -67337485/46017752461    ];
	double* h3bv8=new double[7 ];
	h3=h2*h ;
	h3bv8[0]=0.0;
	h3bv8[1]=h3*1.0/750.0 ;
	h3bv8[2]=h3*(-9.0/16000.0)    ;
	h3bv8[3]=h3* 64.0/5625.0   ;
	h3bv8[4]=h3*  4635333.0/57807374.0   ;	
	h3bv8[5]=h3*  11257698.0/122298911.0    ;	
	h3bv8[6]=h3* (-67337485.0/46017752461.0)   ;	


//% A: c_i
// A = [1/5; 3.0e-01; 8.0e-01 ; 8.0/9.0  ;  1.0];
	double* hA = new double[6 ];
	hA[0]=0.0;
	hA[1]=1.0/5.0*h;
	hA[2]=3.0e-01*h;
	hA[3]=8.0e-01*h;
	hA[4]=8.0/9.0*h;	
	hA[5]=h;

	double**fnewRes=new double*[3];
	fnewRes[0]=new double[nbcmpP1];
	fnewRes[1]=new double[nbcmpP1];
	fnewRes[2]=new double[nbcmpP1];	

	double** col1_6 = new double*[7];
	double* col7   = new double[nbcmpP1];
	double* col8   = new double[nbcmpP1];

	col1_6[0]=new double[nbcmpP1];
	for(int jj=1;jj<=nbcmp;jj++)
		col1_6[0][jj]=0.0;

	for(int i=1;i<=6;i++)
		col1_6[i]=new double[nbcmpP1];

	double eps = 2.220446049250313e-16 ;
	double hmin = 16*eps  ;
	
	int jpnbpos=j+nbpos;

	for(int jj=1;jj<=nbcmp;jj++){
		col1_6[6 ][jj]=fvals[jpnbpos][jj];
		col7[jj]=f2vals[jpnbpos][jj];
		col8[jj]=f3vals[jpnbpos][jj];		
	}
	
	double*tempV=new double[nbcmpP1];
//      yc2 = tempV[i]	
	for(int i=1;i<=nbcmp;i++){
		
		tempV[i]=y[i]+h*(col1_6[6][i]*BMAT[6][1])+col7[i]*h2bv7[1]+col8[i]*h3bv8[1] ;
		}
			
	f(t+hA[1],tempV,nbcmp,col1_6[5]);
	
//      yc3 = tempV[i]	
	for(int i=1;i<=nbcmp;i++){
		
		tempV[i]=y[i]+h*(col1_6[5][i]*BMAT[5][2]+col1_6[6][i]*BMAT[6][2]) +col7[i]*h2bv7[2] +col8[i]*h3bv8[2];
		}
		
	f(t+hA[2],tempV,nbcmp,col1_6[4]);

//      yc4 = tempV[i]	
	for(int i=1;i<=nbcmp;i++)
		tempV[i]=y[i]+h*(col1_6[4][i]*BMAT[4][3] +col1_6[5][i]*BMAT[5][3]+col1_6[6][i]*BMAT[6][3])+col7[i]*h2bv7[3]+col8[i]*h3bv8[3];
			
	f(t+hA[3],tempV,nbcmp,col1_6[3]);

//      yc5 = tempV[i]	
	for(int i=1;i<=nbcmp;i++)
		tempV[i]=y[i]+h*(col1_6[3][i]*BMAT[3][4] +col1_6[4][i]*BMAT[4][4] +col1_6[5][i]*BMAT[5][4]+col1_6[6][i]*BMAT[6][4])+col7[i]*h2bv7[4]+col8[i]*h3bv8[4];		
	f(t+hA[4],tempV,nbcmp,col1_6[2]);

//      yc6 = tempV[i]	
	for(int i=1;i<=nbcmp;i++)
		tempV[i]=y[i]+h*(col1_6[2][i]*BMAT[2][5] +col1_6[3][i]*BMAT[3][5] +col1_6[4][i]*BMAT[4][5] +col1_6[5][i]*BMAT[5][5]+col1_6[6][i]*BMAT[6][5])+col7[i]*h2bv7[5]+col8[i]*h3bv8[5];				
	f(t+hA[5],tempV,nbcmp,col1_6[1]);
	

	double*ynew=new double[nbcmpP1];
	for(int i=1;i<=nbcmp;i++)
		ynew[i]=y[i]+h*(col1_6[1][i]*BMAT[1][6] +col1_6[2][i]*BMAT[2][6] +col1_6[3][i]*BMAT[3][6] +col1_6[4][i]*BMAT[4][6] +col1_6[5][i]*BMAT[5][6]+col1_6[6][i]*BMAT[6][6])+col7[i]*h2bv7[6]+col8[i]*h3bv8[6];
			
	double tnew = t + h ;

	f(tnew, ynew,nbcmp,fnewRes[0]) ;
	for(int jj=1;jj<=nbcmp;jj++)
		col1_6[1][jj]=fnewRes[0][jj] ;
		
	double erk=0.0;
	
//    erk = max(abs(h* col1_6 * Ebv1_6 + h*h *col7 * Ebv7 ))   ;
	erk= -1111.11 ;
	for(int i=1;i<=nbcmp;i++){
        erki= fabs(h*(col1_6[1][i]*Ebv1_6[1]+col1_6[2][i]*Ebv1_6[2]+col1_6[3][i]*Ebv1_6[3]+col1_6[4][i]*Ebv1_6[4]+col1_6[5][i]*Ebv1_6[5]+col1_6[6][i]*Ebv1_6[6])+col7[i]*h2*Ebv7 +col8[i]*h3*Ebv8 ) ; 
        if (erk < erki) erk= erki ;
		}
//	    end for(int i=1;i<=nbcmp;i++)
		
	//    hmin:  
	//    eps = 2.220446049250313e-16 ;
	//    hmin = 16*eps*fabs(tc);    
	//  PART :  STEP SIZE ********************
	
	//  (step size formula : h =hold, stsize = hnew)
	double stsize;
	if (erk != 0.0){
		
		//     ( step cntrl)
		
		stsize = h *stfac *pow(TOL/erk  ,powD  );
		
	}else{
		stsize = 2*h ;
	} 
	//  (end of step size formula : h =hold, stsize = hnew)
	
	//  constraints that stsize must satisfy :
	//     (a) h( next step) = max (h( next step), hmin, 0.5* h(last step))
	//     (b) h( next step) = min (h( next step), hmax, 4 * h(last step))
	
	
	double hmax = 10.0 ;
	
	stsize = max ( stsize, 0.5* h ) ;
	stsize = max ( stsize, hmin ) ;
	stsize = min( stsize, hmax) ;
	stsize = min( stsize, 4*h) ;
	
	resHNEWandERK[0]=stsize;
	resHNEWandERK[1]=erk;

	//freeing memory
	delete Ebv1_6;
	for(int i=0;i<=6;i++)
		delete BMAT[i];
	delete BMAT;
	delete h2bv7;
	delete h3bv8;
	
	delete hA;
	for(int i=0;i<=2;i++)
		delete fnewRes[i];
	delete fnewRes;
	for(int i=0;i<=6;i++)
		delete col1_6[i];
	delete col1_6;
	delete col7;
	delete col8;	
	delete tempV;
	delete ynew;
}

// %%%%%%%%%%%%%%%%%%%%%%%%%%% HBO7inicnstps:   :  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
static void HBO7inicnstps(double RELTOL, double TOL, double t0, double tend, double*y0, int nbinitst, double*tvals, double**yvals, double**fvals, int*res,double**f2vals,double**f3vals, double*resJandH,int nbcmp){

//   subroutine HBO7inicnstps integrates nbinitst (nbinitst = 4) initial constant steps

    double h2,h3, erki, nstsize;
 	int j;
	int nbcmpP1=nbcmp+1;
	double f0[nbcmpP1],f20[nbcmpP1],f30[nbcmpP1] ;
	
	//------- PARAM: -------
	// eps of hmin:  eps = 2.220446049250313e-16
	double eps = 2.220446049250313e-16 ;
	double hmax = 0.1*fabs(tend-t0)  ;
	double stfac=0.81 ;
	double hv;
		
	int nbeps=8 ;
	double hmin = 16*eps ;
	
	//------- END PARAM ----
		
	int nbpos=10;
	

	double TOL111=TOL/500;
		
	double tc = t0;
	double* yc =new double[nbcmpP1];
	for(int i=1;i<=nbcmp;i++)
		yc[i]=y0[i];
	
	double**fcRes=new double*[3];
	fcRes[0]=new double[nbcmpP1];
	fcRes[1]=new double[nbcmpP1];
	fcRes[2]=new double[nbcmpP1];	
	fpp(tc,yc,nbcmp,fcRes);
	for(int i=1;i<=nbcmp;i++){
		f0[i] = fcRes[0][i];
		f20[i] = fcRes[1][i];
		f30[i] = fcRes[2][i];		
		} 
	
	// p 169, Hairer: first guess for step size:
	double sc=0.0;double d1=0.0;
	for(int i=1;i<=nbcmp;i++){
		sc+=pow(y0[i],2.0);
		d1+=pow(fcRes[0][i],2.0);
	}
	double d0=sqrt(sc);
	sc=sqrt(sc)*RELTOL+TOL111;
	double sc2=sqrt(nbcmp*1.0)*sc;
	//  d0 = norm(y0)/(sqrt(nbcmp)*sc)
	d0/=sc2;
	//  d0 = norm(y0);
	//  d1 = norm(fc)/(sqrt(nbcmp)*sc)
	d1=sqrt(d1)/sc2;
	double h0;
	//   d1 = norm(fc) ;
	
	if ((d0 < 0.00001)||(d1 < 0.00001))
		h0= 1.0e-6;
	else
		h0=0.01*d0/d1;

	//   (if (d0 < 0.00001)||(d1 < 0.00001) )
	// perform one expl. Euler step
	double*y1=new double[nbcmpP1];
	for(int i=1;i<=nbcmp;i++)
        y1[i]=y0[i]+h0*fcRes[0][i];
	double*fc1=new double[nbcmpP1];
	f(tc+h0,y1,nbcmp,fc1);

	// compute d2 = p168norm(fc1-fc)/h0
	double d2=0;
	for(int i=1;i<=nbcmp;i++)
        d2+=pow(fc1[i]-fcRes[0][i],2.0);
	d2=sqrt(d2)/sc2/h0;
	
	// e) p169: (HBT7  order p=7)
	double h1=pow(0.01/max(d1,d2) , 1.0/5.0) ;
	
	if (max(d1,d2) <=pow( 10.0, -15.0) ){
		h1=max(pow(10.0, -6.0), h0*pow(10.0, -3.0));
	}
	
	double h=min(100.0*h0, h1) ;
	
//	cout <<  "HBO7inicnstps: nbinitst" << nbinitst  << " p169h = " <<  h  << endl ;		
	
	int k=1111 ;		
	int iter=0;	
	double erk=1111.11 ;
	double hold;
	double*resHNEWandERK=new double[2];
	double hnew;
	
	j=0 ;
	int nreje=0;
	
	// init param:
	double powD = 1.0/6.0;
	
	// E = coeffs(sol y) - coeffs(SCP y)
// HBO7
// Ebv1_6 = [ 3062780599/29816473381-581429233/7480971313 ;  -1533181/7306695+9222278/50999049  ;  605283937/1107746875-4539313/7992920 ; 41554493/75355849-3692668/11716025  ; 0 ;  720961/77876101-14791056/67227925 ] ; 
// Ebv7   = [-27007585/1439293967-1563392/45805487 ] ;
// Ebv8   = [ -67337485/46017752461-1015453/304980503 ];

	double* Ebv1_6 = new double[7];
	Ebv1_6[1]= 3062780599.0/29816473381.0 -581429233.0/7480971313.0     ;
	Ebv1_6[2]= -1533181.0/7306695.0 +9222278.0/50999049.0   ;
	Ebv1_6[3]= 605283937.0/1107746875.0 -4539313.0/7992920.0               ;
	Ebv1_6[4]= 41554493.0/75355849.0 -3692668.0/11716025.0         ;
	Ebv1_6[5]= 0 ;
	Ebv1_6[6]= 720961.0/77876101.0-14791056.0/67227925.0  ;		

	double Ebv7   = -27007585.0/1439293967.0-1563392.0/45805487.0    ;
	double Ebv8   = -67337485.0/46017752461.0-1015453.0/304980503.0 ;

	double** BMAT=new double*[7 ];
	for(int i=0;i<=6;i++){
		BMAT[i]=new double[7 ];
	}

//% *** HBO7 B *****************
//% HBO7
//% c2    c3       c4             c5                    c6                    IF
//B=[
//   0     0        0              0                     0                3062780599/29816473381
//   0     0        0              0              -9749783/42956078       -1533181/7306695
//   0     0        0         -7617722/20101141   3327355531/11484129947   605283937/1107746875 
//   0     0     16384/2025    57594571/1807044    86767468/3359647        41554493/75355849 
//   0    81/320  -1088/75     -26032617/407999     -34264951556/628358387            0
//  1/5   3/64   14612/2025   150112116/4521287    83898588/2830417        720961/77876101  ];
//bv7= [ 1/50  ;  -9/1600 ;  536/675 ;  14035033/3600870 ;  28792462/7935665 ;   -27007585/1439293967 ];
//bv8= [ 1/750 ;  -9/16000 ; 64/5625 ;  4635333/57807374 ;  11257698/122298911 ; -67337485/46017752461    ];
	
	BMAT[1][6]=3062780599.0/29816473381.0  ;
	
	BMAT[2][5]= -9749783.0/42956078.0   ;
	BMAT[2][6]=  -1533181.0/7306695.0   ;
	
	BMAT[3][4]= -7617722.0/20101141.0 ;
	BMAT[3][5]=  3327355531.0/11484129947.0   ;
	BMAT[3][6]=  605283937.0/1107746875.0   ;

	BMAT[4][3]=  16384.0/2025.0     ;
	BMAT[4][4]=  57594571.0/1807044.0    ;
	BMAT[4][5]=  86767468.0/3359647.0   ;
	BMAT[4][6]=  41554493.0/75355849.0    ;

	BMAT[5][2]=  81.0/320.0     ;
	BMAT[5][3]=  -1088.0/75.0      ;
	BMAT[5][4]=  -26032617.0/407999.0   ;
	BMAT[5][5]=   -34264951556.0/628358387.0      ;
	BMAT[5][6]=  0.0  ;	

//  1/5   3/64   14612/2025   150112116/4521287    83898588/2830417        720961/77876101
	BMAT[6][1]=  1.0/5.0  ;
	BMAT[6][2]=  3.0/64.0   ;
	BMAT[6][3]=  14612.0/2025.0     ;
	BMAT[6][4]=  150112116.0/4521287.0    ;
	BMAT[6][5]=  83898588.0/2830417.0    ;
	BMAT[6][6]=  720961.0/77876101.0  ;	

	double** col1_6 = new double*[7];
	double* col7   = new double[nbcmpP1];
	double* col8   = new double[nbcmpP1];
	
	for(int i=0;i<=6;i++){
		col1_6[i]=new double[nbcmpP1];
		for(int jj=1;jj<=nbcmp;jj++)
			col1_6[i][jj]=0.0;
	}
	for(int jj=1;jj<=nbcmp;jj++)
		col7[jj]=0.0;
			
	double* h2bv7=new double[7 ];
	double* h3bv8=new double[7 ];

	h2bv7[0]=0.0;
	h3bv8[0]=0.0;
		
	double* hA = new double[6 ];
	hA[0]=0.0;
	double*ynew=new double[nbcmpP1];
	double**fnewRes=new double*[3];
	fnewRes[0]=new double[nbcmpP1];
	fnewRes[1]=new double[nbcmpP1];
	fnewRes[2]=new double[nbcmpP1];
		
	double*tempV=new double[nbcmpP1];

	double tempD;
	double tnew;
	double stsize;
	double prevh;
	int jpnbposp1;
	bool Nret=true;
	// ----------- iterations until fabs(erk) <= TOL111 --------------	
	while(  fabs(erk) > TOL111){
		hold=h;
	   j=0 ;
	   int jpnbpos=j+nbpos;
	   tvals[jpnbpos] =  t0;
	   for(int i=1;i<=nbcmp;i++){
	   	   yvals[jpnbpos][i] =  y0[i];
		   fvals[jpnbpos][i] =  f0[i] ;
		   f2vals[jpnbpos][i] = f20[i] ;
		   f3vals[jpnbpos][i] = f30[i] ;	
	   	   yc[i] =  y0[i];	
		   	   	
	   }
	   // ( end for(int i=1;i<=nbcmp;i++)
	   tc=t0;	   										
	   iter++;
//  act 	hv=h ;	
	hv=h ;
//   set hv: const step size )) 

	// ( HBO7d123initd1d2steps.m MAIN LOOP  1 integration step/ iteration)		
    // (nbinitst=4)
	while(  j  < nbinitst && Nret){
		
		if( j >= 1000000  ){

	        cout << "HBO7inisteps:  j= " << j << ", hm8 =" << tvals[j-7+nbpos]-tvals[j-8+nbpos]  << endl ;
	        cout << "  j= " << j << ", hm7 =" << tvals[j-6+nbpos]-tvals[j-7+nbpos]  << endl ;
	        cout << "  j= " << j << ", hm6 =" << tvals[j-5+nbpos]-tvals[j-6+nbpos]  << endl ;
	        cout << "  j= " << j << ", hm5 =" << tvals[j-4+nbpos]-tvals[j-5+nbpos]  << endl ;
	        cout << "  j= " << j << ", hm4 =" << tvals[j-3+nbpos]-tvals[j-4+nbpos]  << endl ;
	        cout << "  j= " << j << ", hm3 =" << tvals[j-2+nbpos]-tvals[j-3+nbpos]  << endl ;
	        cout << "  j= " << j << ", hm2 =" << tvals[j-1+nbpos]-tvals[j-2+nbpos]  << endl ;
	        cout << "  j= " << j << ", hm1 =" << tvals[j+nbpos]-tvals[j-1+nbpos]  << endl ;
	        cout << "  tvals= " << tvals[j+nbpos ]  << endl ;
	        cout << "  yvalszero= " << yvals[j+nbpos ][1]-exp(-tvals[j+nbpos ]) << endl ;
			
			cout<<"Debug stop.";
			int err;
//			cin>>err;
			Nret=false;
			exit(1) ;			
		}

//      const h:
        h=hv ;		
//		h=hnew;		
		//  (MAIN INTEG STEP)		
		//    t=tc;
		jpnbpos=j+nbpos;

	for(int jj=1;jj<=nbcmp;jj++){
		col1_6[6 ][jj]=fvals[jpnbpos][jj];
		col7[jj]=f2vals[jpnbpos][jj];
		col8[jj]=f3vals[jpnbpos][jj];		
	}
	
	hA[0]=0.0;
	hA[1]=1.0/5.0*h;
	hA[2]=3.0e-01*h;
	hA[3]=8.0e-01*h;
	hA[4]=8.0/9.0*h;	
	hA[5]=h;

	h2=h*h ;
	h2bv7[0]=0.0;
	h2bv7[1]=h2*1.0/50.0 ;
	h2bv7[2]=h2* (-9.0/1600.0)  ;
	h2bv7[3]=h2* 536.0/675.0  ;
	h2bv7[4]=h2*  14035033.0/3600870.0  ;	
	h2bv7[5]=h2* 28792462.0/7935665.0   ;	
	h2bv7[6]=h2*  (-27007585.0/1439293967.0)  ;

	h3=h2*h ;
	h3bv8[0]=0.0;
	h3bv8[1]=h3*1.0/750.0 ;
	h3bv8[2]=h3*(-9.0/16000.0)    ;
	h3bv8[3]=h3* 64.0/5625.0   ;
	h3bv8[4]=h3*  4635333.0/57807374.0   ;	
	h3bv8[5]=h3*  11257698.0/122298911.0    ;	
	h3bv8[6]=h3* (-67337485.0/46017752461.0)   ;	

//      yc2 = tempV[i]	
	for(int i=1;i<=nbcmp;i++)
		tempV[i]=yc[i]+h*(col1_6[6][i]*BMAT[6][1])+col7[i]*h2bv7[1] +col8[i]*h3bv8[1] ;		
	f(tc+hA[1],tempV,nbcmp,col1_6[5]);
	
//      yc3 = tempV[i]	
	for(int i=1;i<=nbcmp;i++)
		tempV[i]=yc[i]+h*(col1_6[5][i]*BMAT[5][2]+col1_6[6][i]*BMAT[6][2])+col7[i]*h2bv7[2]+col8[i]*h3bv8[2] ;			
	f(tc+hA[2],tempV,nbcmp,col1_6[4]);

//      yc4 = tempV[i]	
	for(int i=1;i<=nbcmp;i++)
		tempV[i]=yc[i]+h*(col1_6[4][i]*BMAT[4][3] +col1_6[5][i]*BMAT[5][3]+col1_6[6][i]*BMAT[6][3])+col7[i]*h2bv7[3]+col8[i]*h3bv8[3] ;		
	f(tc+hA[3],tempV,nbcmp,col1_6[3]);

//      yc5 = tempV[i]	
	for(int i=1;i<=nbcmp;i++)
		tempV[i]=yc[i]+h*(col1_6[3][i]*BMAT[3][4] +col1_6[4][i]*BMAT[4][4] +col1_6[5][i]*BMAT[5][4]+col1_6[6][i]*BMAT[6][4])+col7[i]*h2bv7[4] +col8[i]*h3bv8[4];		
	f(tc+hA[4],tempV,nbcmp,col1_6[2]);

//      yc6 = tempV[i]	
	for(int i=1;i<=nbcmp;i++)
		tempV[i]=yc[i]+h*(col1_6[2][i]*BMAT[2][5] +col1_6[3][i]*BMAT[3][5] +col1_6[4][i]*BMAT[4][5] +col1_6[5][i]*BMAT[5][5]+col1_6[6][i]*BMAT[6][5])+col7[i]*h2bv7[5]+col8[i]*h3bv8[5];			
	f(tc+hA[5],tempV,nbcmp,col1_6[1]);
	

	for(int i=1;i<=nbcmp;i++)
		ynew[i]=yc[i]+h*(col1_6[1][i]*BMAT[1][6] +col1_6[2][i]*BMAT[2][6] +col1_6[3][i]*BMAT[3][6] +col1_6[4][i]*BMAT[4][6] +col1_6[5][i]*BMAT[5][6]+col1_6[6][i]*BMAT[6][6])+col7[i]*h2bv7[6] +col8[i]*h3bv8[6];
			
	tnew = tc + h ;

		fpp(tnew, ynew,nbcmp,fnewRes) ;
		for(int jj=1;jj<=nbcmp;jj++)
			col1_6[1][jj]=fnewRes[0][jj] ;

	 erk=0.0;

//    erk = max(abs(h* col1_6 * Ebv1_6 + h*h *col7 * Ebv7 ))   ;
	erk= -1111.11 ;
	for(int i=1;i<=nbcmp;i++){
        erki= fabs(h*(col1_6[1][i]*Ebv1_6[1]+col1_6[2][i]*Ebv1_6[2]+col1_6[3][i]*Ebv1_6[3]+col1_6[4][i]*Ebv1_6[4]+col1_6[5][i]*Ebv1_6[5]+col1_6[6][i]*Ebv1_6[6])+col7[i]*h2*Ebv7 +col8[i]*h3*Ebv8 ) ; 
        if (erk < erki) erk= erki ;
		}
//	    end for(int i=1;i<=nbcmp;i++)
				
		tc=tnew;
		jpnbposp1=j+nbpos+1;
			
		for(int i=1;i<=nbcmp;i++)	{
			yc[i]=ynew[i];
			yvals[jpnbposp1][i] =  yc[i];
			fvals[jpnbposp1][i]=  fnewRes[0][i] ;
			f2vals[jpnbposp1][i]=  fnewRes[1][i] ;
			f3vals[jpnbposp1][i]=  fnewRes[2][i] ;			
		}
		
		//  -----(END OF INTEG STEP) --------------------			
		//  (yvals,tvals,.. UPDATE STEP)
		
		tvals[jpnbposp1] =  tc;
	
		prevh = h ;
		j++;				
	}
	// ( END OF HBO7d123initstepsprg.m MAIN LOOP  1 integration step/ iteration)

    if( fabs(erk) > TOL111 ){
	
		//    hmin: 
		//    eps = 2.220446049250313e-16 ;
		//    hmin = 16*eps*fabs(tc);    stephmin2=hmin
		// ***** (( PART :  STEP SIZE ********************				
		//  (step size formula : h =hold, stsize = hnew)
		
		if (erk != 0.0){
			
			// ACT    (HB../HB11 step cntrl)
			stsize = 0.7*h *stfac *pow((TOL/500.0)/erk  ,powD  );
			
		}else{
			stsize = 1.9 *h ;
		} 
		//  (end of step size formula : h =hold, stsize = hnew)
		
		//  constraints that stsize must satisfy :
		//     (a) h( next step) = max (h( next step), hmin, 0.5* h(last step))
		//     (b) h( next step) = min (h( next step), hmax, 4 * h(last step))
				
		stsize = max ( stsize, 0.24* h ) ;
		stsize = max ( stsize, hmin ) ;
		stsize = min( stsize, hmax) ;
		stsize = min( stsize, 4*h) ;	
		
		h=stsize ;			
		
		// *****  PART :  STEP SIZE )) ********************		
		}						
//      (end if( fabs(erk) > TOL111 ){  )

	}
//	( end while(  fabs(erk) > TOL111){ )
	
	// print out statistics
	// with index j-1,  yvals(n+1+nbpos, . )  cont solution
	int n=j-1;
	int nb_f_eval=(n+1+nreje)*nbeps ;
		
	int NFE=nb_f_eval;
	
	res[1]=NFE;
	// nb succes step:
	res[2]=n+1 ;
	res[3]=nreje ;

	resJandH[0]=j;
	resJandH[1]=h;

	//freeing memory
	delete yc;
	for(int i=0;i<=1;i++)
		delete fcRes[i];
	delete fcRes;
	delete y1;
	delete fc1;
	delete resHNEWandERK;
	delete Ebv1_6;
	for(int i=0;i<=6;i++)
		delete col1_6[i];
	delete col1_6;	
	delete col7;
	delete col8;	
	for(int i=0;i<=6;i++)
		delete BMAT[i];
	delete BMAT;
	delete h2bv7;
	delete h3bv8;	
	delete hA;
	for(int i=0;i<=2;i++)
		delete fnewRes[i];
	delete fnewRes;
	delete tempV;
	delete ynew;

}
//  END OF HBO7d123initcnststeps.m ********************


//id123O14 output :  coeffs of interpol polyn  d123,
// so that the value of interpol polyn at t=tend, err O(h^15) can be obtained %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

static void id123O14(double*xtab,double*b_v){

	double xpos = xtab[1] ;	
	double xkm11= xtab[2] ;
	double xkm10= xtab[3] ;
	double xkm9 = xtab[4] ;
	double xkm8 = xtab[5] ;
	double xkm7 = xtab[6] ;
	double xkm6 = xtab[7] ;
	double xkm5 = xtab[8] ;
	double xkm4 = xtab[9] ;
	double xkm3 = xtab[10] ;
	double xkm2 = xtab[11] ;
	double xkm1 = xtab[12] ;
	double xk1  = xtab[13] ;
	double xkp1  = xtab[14] ;

	int n=14 ;	
	
	double hl = (xkp1-xk1) ;
	
	double hm11= (xkm11-xk1)/hl;
	double hm10= (xkm10-xk1)/hl;
	double hm9 = (xkm9-xk1)/hl;
	double hm8 = (xkm8-xk1)/hl;
	double hm7 = (xkm7-xk1)/hl;
	double hm6 = (xkm6-xk1)/hl;
	double hm5 = (xkm5-xk1)/hl;
	double hm4 = (xkm4-xk1)/hl;
	double hm3 = (xkm3-xk1)/hl;
	double hm2 = (xkm2-xk1)/hl;
	double hm1 = (xkm1-xk1)/hl;
	double hp1 = (xkp1-xk1)/hl;
	double hin = (xpos-xk1)/hl;	
		
	double* ft=new double[18];
	ft[1]=1.0;
	ft[2]=2.0;
	ft[3]=6.0;
	ft[4]=24.0;
	ft[5]=120.0;
	ft[6]=720.0;
	ft[7]=5040.0;
	ft[8]=40320.0;
	ft[9]=362880.0;
	ft[10]=3628800.0;
	ft[11]=39916800.0;
	ft[12]=479001600.0;
	ft[13]=6.227020800000000e+09;
	ft[14]=8.717829120000000e+10;
	ft[15]=1.307674368000000e+12;
	ft[16]=2.092278988800000e+13;
	ft[17]=3.556874280960000e+14;
	
	double rift2 =     5.000000000000000e-01 ;
	double rift3 =     1.666666666666667e-01 ;
	double rift4 =     4.166666666666666e-02 ;
	double rift5 =     8.333333333333333e-03 ;
	double rift6 =     1.388888888888889e-03 ;
	double rift7 =     1.984126984126984e-04 ;
	double rift8 =     2.480158730158730e-05 ;
	double rift9 =     2.755731922398589e-06 ;
	double rift10 =     2.755731922398589e-07;
	double rift11 =     2.505210838544172e-08;
	double rift12 =     2.087675698786810e-09;
	double rift13 =     1.605904383682161e-10;
	double rift14 =     1.147074559772972e-11;
	double rift15 =     7.647163731819816e-13;
	double rift16 =     4.779477332387385e-14;
	double rift17 =     2.811457254345521e-15 ;

    double    hin2= hin*hin/2 ;
    double    hin3= hin2 *hin/3  ;
    double    hin4= hin3 *hin/4  ;
    double    hin5= hin4 *hin/5  ;
    double    hin6= hin5 *hin/6  ;
    double    hin7= hin6 *hin/7  ;
    double    hin8= hin7 *hin/8  ;
    double    hin9= hin8 *hin/9  ;
    double    hin10=hin9 *hin/10  ;
    double    hin11=hin10*hin/11  ;
    double    hin12=hin11*hin/12  ;
    double    hin13=hin12*hin/13  ;
    double    hin14=hin13*hin/14  ;
    double    hin15=hin14*hin/15  ;
    double    hin16=hin15*hin/16  ;
	
	double* A_MAT2col=new double[20];
//  A_MAT2col[0-3]= nil ;	
	A_MAT2col[0] =0.0 ;
	A_MAT2col[1] =0.0 ;
	A_MAT2col[2] =0.0 ;
	A_MAT2col[3] =111.1 ;

	A_MAT2col[4] =hp1 ;
	A_MAT2col[5] =hp1 ;
	A_MAT2col[6] =hp1 ;
	A_MAT2col[7] =hm1 ;
	A_MAT2col[8] =hm1 ;
	A_MAT2col[9] =hm1 ;
	A_MAT2col[10]=hm2 ;
	A_MAT2col[11]=hm2 ;
	A_MAT2col[12]=hm2 ;
	A_MAT2col[13]=hm3 ;
	A_MAT2col[14]=hm3 ;
//  A_MAT2col[15-19] = na	
	A_MAT2col[15] =0.0 ;
	A_MAT2col[16] =0.0 ;
	A_MAT2col[17] =0.0 ;
	A_MAT2col[18] =0.0 ;
	A_MAT2col[19] =0.0 ;
	
	double* RHS=new double[20];
	RHS[0]  =0.0;
	RHS[1]  =hin ;
	RHS[2]  =hin2  ;
	RHS[3]  =hin3  ; 
	RHS[4]  =hin4  ;
	RHS[5]  =hin5  ;
	RHS[6]  =hin6 ;
	RHS[7]  =hin7 ;
	RHS[8]  =hin8 ;
	RHS[9]  =hin9 ;
	RHS[10] =hin10 ;
	RHS[11] =hin11 ;
	RHS[12] =hin12 ;
	RHS[13] =hin13 ;
	RHS[14] =hin14 ;
	RHS[15]  =0.0;
	RHS[16]  =0.0;
	RHS[17]  =0.0;
	RHS[18]  =0.0;
	RHS[19]  =0.0;
	
	// ----------- ALGORITHM ---------------------------------
	double*invdiag=new double[20];
	for( int idx=n+1;idx<=19;idx++){
		invdiag[idx]=0.0;
	}

	for( int idx=1;idx<=n;idx++){
		invdiag[idx]=1.0;
	}
	double multip;
	for( int idx=4;idx<n;idx++){
		multip =-A_MAT2col[idx]/idx ;
		invdiag[idx+1]= invdiag[idx]*multip  ;
		
	}
	//      (end for idx=4:n)
	double** lop=new double*[20];
	for(int i=0;i<=19;i++){	
		lop[i]=new double[20];
		for(int jj=1;jj<=19;jj++)
			lop[i][jj]=0.0;
	}

//	cout   << "interpol: lop " <<  endl ;			
	lop[4][5]= -4.0/hp1;
	lop[4][6]= -5.0/hp1;
	lop[4][7]= -6.0/hp1;
	lop[4][8]= -7.0/hp1;
	lop[4][9]= -8.0/hp1;
	lop[4][10]=-9.0/hp1;
	lop[4][11]=-10.0/hp1;
	lop[4][12]=-11.0/hp1;
	lop[4][13]=-12.0/hp1;
	lop[4][14]=-13.0/hp1;
		
	lop[5][6]=  -4.0/hp1;
	lop[5][7]=  -5.0/hp1;
	lop[5][8]=  -6.0/hp1;
	lop[5][9]=  -7.0/hp1;
	lop[5][10]= -8.0/hp1;
	lop[5][11]= -9.0/hp1;
	lop[5][12]=-10.0/hp1;
	lop[5][13]=-11.0/hp1;
	lop[5][14]=-12.0/hp1;

	lop[6][7]=  -4.0/hp1;
	lop[6][8]=  -5.0/hp1;
	lop[6][9]=  -6.0/hp1;
	lop[6][10]= -7.0/hp1;
	lop[6][11]= -8.0/hp1;
	lop[6][12]= -9.0/hp1;
	lop[6][13]=-10.0/hp1;
	lop[6][14]=-11.0/hp1;
	
	lop[7][8]=  -4.0/hm1;
	lop[7][9]=  -5.0/hm1;
	lop[7][10]= -6.0/hm1;
	lop[7][11]= -7.0/hm1;
	lop[7][12]= -8.0/hm1;
	lop[7][13]= -9.0/hm1;
	lop[7][14]=-10.0/hm1;
	
	lop[8][9]= -4.0/hm1;
	lop[8][10]=-5.0/hm1;
	lop[8][11]=-6.0/hm1;
	lop[8][12]=-7.0/hm1;
	lop[8][13]=-8.0/hm1;
	lop[8][14]=-9.0/hm1;	
	
	lop[9][10]=-4.0/hm1;
	lop[9][11]=-5.0/hm1;
	lop[9][12]=-6.0/hm1;
	lop[9][13]=-7.0/hm1;
	lop[9][14]=-8.0/hm1;

	lop[10][11]=-4.0/hm2;
	lop[10][12]=-5.0/hm2;
	lop[10][13]=-6.0/hm2;
	lop[10][14]=-7.0/hm2;
	
	lop[11][12]=-4.0/hm2;
	lop[11][13]=-5.0/hm2;
	lop[11][14]=-6.0/hm2;
		
	lop[12][13]=-4.0/hm2;
	lop[12][14]=-5.0/hm2;
	
	lop[13][14]=-4.0/hm3;
	
	//-----Uoper.------------
		
	double** U=new double*[20];
	for(int i=0;i<=19;i++){	
		U[i]=new double[20];
		for(int jj=1;jj<=19;jj++)
			U[i][jj]=0.0;
	}
	
	//specialU1:
	
	U[1][4]=1.0/hp1 ;
	U[1][7]=1.0/(hm1-hp1);
	U[1][10]=1.0/(hm2-hm1);
	U[1][13]=1.0/(hm3-hm2);

	U[2][4]=2.0/hp1;
	U[2][5]=2.0/hp1;
	U[2][7]=2.0/(hm1-hp1);
	U[2][8]=2.0/(hm1-hp1);
	U[2][10]=2.0/(hm2-hm1);
	U[2][11]=2.0/(hm2-hm1);
	U[2][13]=2.0/(hm3-hm2);
	U[2][14]=2.0/(hm3-hm2);
	
	U[3][4]=3.0/hp1;
	U[3][5]=3.0/hp1;
	U[3][6]=3.0/hp1;
	U[3][7]=3.0/(hm1-hp1);
	U[3][8]=3.0/(hm1-hp1);
	U[3][9]=3.0/(hm1-hp1);
	U[3][10]=3.0/(hm2-hm1);
	U[3][11]=3.0/(hm2-hm1);
	U[3][12]=3.0/(hm2-hm1);
	U[3][13]=3.0/(hm3-hm2);
	U[3][14]=3.0/(hm3-hm2);
	
	U[4][5]=4.0/hp1;
	U[4][6]=4.0/hp1;
	U[4][7]=4.0/hm1;
	U[4][8]=4.0/(hm1-hp1);
	U[4][9]=4.0/(hm1-hp1);
	U[4][10]=4.0/(hm2-hp1);
	U[4][11]=4.0/(hm2-hm1);
	U[4][12]=4.0/(hm2-hm1);
	U[4][13]=4.0/(hm3-hm1);
	U[4][14]=4.0/(hm3-hm2);

	U[5][6]=5.0/hp1;
	U[5][7]=5.0/hm1;
	U[5][8]=5.0/hm1;
	U[5][9]=5.0/(hm1-hp1);
	U[5][10]=5.0/(hm2-hp1);
	U[5][11]=5.0/(hm2-hp1);
	U[5][12]=5.0/(hm2-hm1);
	U[5][13]=5.0/(hm3-hm1);
	U[5][14]=5.0/(hm3-hm1);
	
	U[6][7]=6.0/hm1;
	U[6][8]=6.0/hm1;
	U[6][9]=6.0/hm1;
	U[6][10]=6.0/(hm2-hp1);
	U[6][11]=6.0/(hm2-hp1);
	U[6][12]=6.0/(hm2-hp1);
	U[6][13]=6.0/(hm3-hm1);
	U[6][14]=6.0/(hm3-hm1);
		
	U[7][8]=7.0/hm1;
	U[7][9]=7.0/hm1;
	U[7][10]=7.0/hm2;
	U[7][11]=7.0/(hm2-hp1);
	U[7][12]=7.0/(hm2-hp1);
	U[7][13]=7.0/(hm3-hp1);
	U[7][14]=7.0/(hm3-hm1);
	
	U[8][9]=8.0/hm1;
	U[8][10]=8.0/hm2;
	U[8][11]=8.0/hm2;
	U[8][12]=8.0/(hm2-hp1);
	U[8][13]=8.0/(hm3-hp1);
	U[8][14]=8.0/(hm3-hp1);

	U[9][10]=9.0/hm2;
	U[9][11]=9.0/hm2;
	U[9][12]=9.0/hm2;
	U[9][13]=9.0/(hm3-hp1);
	U[9][14]=9.0/(hm3-hp1);
	
	U[10][11]=10.0/hm2;
	U[10][12]=10.0/hm2;
	U[10][13]=10.0/hm3;
	U[10][14]=10.0/(hm3-hp1);
	
	U[11][12]=11.0/hm2;
	U[11][13]=11.0/hm3;
	U[11][14]=11.0/hm3;
	
	U[12][13]=12.0/hm3;
	U[12][14]=12.0/hm3;
	
	U[13][14]=13.0/hm3;
		

	// ALGORITHM CACUL.: b_v2 = U2*U3*U4*U5*U6*U7*U8*U9*U10*U11*U12*U13*U14*inv(DIAG)*l17 *l16*l15*l14*l13*l12*l11*l10*l9*l8*l7*l6*l5*l3*l2*RHS ;
		
	// --- PART 1: CALCUL.  l13*l12*l11*l10*l9*l8*l7*l6*l5*l3*l2*RHS =bv2()   -------------------------------------------
	
	double*b_v2=new double[20];
	for( int idx=n+1;idx<=19;idx++){
		b_v2[idx]=0.0 ;
	}
	for( int idx=1;idx<=n;idx++){
		b_v2[idx]=RHS[idx] ;
	}
	
	//   k: operation matri l_k
	
	for(  int k=4;k<=n-1;k++){
		for( int idx=n;idx>=k+1;idx--){
			b_v2[ idx]=b_v2[ idx-1]+lop[k][idx]*b_v2[ idx] ;
		}
	}

	// SECTION 2.4: operations inv(DIAG)*b_v2()  = b_v2()
		
	for( int k=1;k<=n;k++){
		b_v2[k]*= invdiag[k];
	}
	//      (end for k=n-1:-1:1)

	
	// SECTION 2.5: operations U*inv(DIAG)*L*RHS() = b_v2() *****************
	
	// subsection U_3 -> U{n-1} *b_v2() =b_v2()	---------------
	for( int k=n-1;k>=3;k--){
		for( int idx=k+1;idx<=n;idx++){
//       	equiv		b_v2[idx] = U[k][ idx]*b_v2[idx] ;		
            b_v2[idx]*= U[k][ idx];
		}
		//        (end for idx=k+1:n-2)
		
		for( int idx=k;idx<=n-1;idx++){
//         equiv			b_v2[idx]=b_v2[idx] - b_v2[ idx+1] ;
 		   b_v2[idx]-=b_v2[ idx+1] ;
		}
		//        (end for i=k:n-1)
	}
	//       (end for k=n-3:-1:2)

	// subsection oper U2*b_v2() = b_v2() ----------------------

  b_v2[4]= U[2][4]* b_v2[4] ;
  b_v2[5]= U[2][5]* b_v2[5] ;
  b_v2[7]= U[2][7]* b_v2[7] ;
  b_v2[8]= U[2][8]* b_v2[8] ;
  b_v2[10]=U[2][10]*b_v2[10] ;
  b_v2[11]=U[2][11]*b_v2[11] ;
  b_v2[13]=U[2][13]*b_v2[13] ;
  b_v2[14]=U[2][14]*b_v2[14] ;

  b_v2[2]= b_v2[ 2]-b_v2[ 4] ;
  b_v2[4]= b_v2[ 4]-b_v2[ 5] ;
  b_v2[5]= b_v2[ 5]-b_v2[ 7] ;
  b_v2[7]= b_v2[ 7]-b_v2[ 8] ;
  b_v2[8]= b_v2[ 8]-b_v2[10] ;
  b_v2[10]=b_v2[10]-b_v2[11] ;
  b_v2[11]=b_v2[11]-b_v2[13] ;
  b_v2[13]=b_v2[13]-b_v2[14] ;
		
	// subsection  oper U1*b_v2() = b_v2() --------------
	
	// p 187 Golub
    // --- Iter PART1:  for idx=4:2:n-1:  oper U(1,4),U(1,7),...U(1,13) ---
	for( int idx=4;idx<=n-1;idx +=3){
       b_v2[idx]= U[1][ idx] *b_v2[idx] ;
       }
    // (end for i=k+1:n)

	// --- Iter PART2: ---
	for( int idx=1;idx<=n-4;idx +=3){
       b_v2[idx]= b_v2[ idx]-b_v2[ idx+3] ;
	   }
    //(end for i=k:n-1)


 b_v[1]=b_v2[4];
 b_v[2]=b_v2[1];
 b_v[3]=b_v2[7];
 b_v[4]=b_v2[10];
 b_v[5]=b_v2[13];
 b_v[6]=b_v2[5];
 b_v[7]=b_v2[2];
 b_v[8]=b_v2[8];
 b_v[9]=b_v2[11];
b_v[10]=b_v2[14];
b_v[11]=b_v2[6];
b_v[12]=b_v2[3];
b_v[13]=b_v2[9];
b_v[14]=b_v2[12];


	//freeing memory
	delete ft;
	delete RHS;
	delete invdiag;
	delete A_MAT2col;
	for(int i=0;i<=19;i++)
		delete lop[i];
	delete lop;
	for(int i=0;i<=19;i++)
		delete U[i];
	delete U;
	delete b_v2;
}
// end HBO14d123interpol %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%	




// %%%%%%%%%%%%%%  HBO144q16 %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
static void HBO144q16(double RELTOL, double TOL, double t0, double tend, double*y0,double*tvals, double**yvals, double**fvals, double**f2vals, double**f3vals, int*res,int nbcmp){

/*

% ************** Table of sections and subsections of "HBO144q16" ******************************
% 
% 
% SECTION 1. INITIALISATION
%     SUBS 1.A  INPUT THE MATRIX OF COEFFICIENTS 
%           file  ""HBO144cfs.cpp"
% 
%     SUBS 1.B THE FIRST FOUR EQUAL CONSECUTIVE STEPSIZES 
%         are obtained by the procedure "HBO7inicnstps"
% 
% SECTION 2. PERFORM THE INTEGRATION : 
%       MAIN LOOP TO ADVANCE INTEGRATION FROM x_n to x_{n+3} PER ITERATION 
%          {
%            (SUBS 2.A) SUBLOOP REPEAT THE FOLLOWING THREE STEPS UNTIL err_est <= TOL
%              {
% 
%               SUBS 2.A.1 STEP1 : TO ADVANCE INTEGRATION FROM x_n to x_{n+1} 
%                   The values y_jpc2,y_jpc3,y_jpc4 and ynew of
%                   predictors P2, P3, P4 and Integration Formula  respectively are obtained 
%                   (y_jpc2,y_jpc3,y_jpc4 and ynew of STEP1)
% 
%               SUBS 2.A.2 STEP2 : TO ADVANCE INTEGRATION FROM x_{n+1} to x_{n+2} 
%                   The values y_jpc2,y_jpc3,y_jpc4 and ynew of
%                   predictors P2, P3, P4 and Integration Formula  respectively are obtained 
%                   (y_jpc2,y_jpc3,y_jpc4 and ynew of STEP2)
% 
%               SUBS 2.A.3 STEP3 : TO ADVANCE INTEGRATION FROM x_{n+2} to x_{n+3} 
%                   The values y_jpc2,y_jpc3,y_jpc4 and ynew of
%                   predictors P2, P3, P4 and Integration Formula  respectively are obtained 
%                   (y_jpc2,y_jpc3,y_jpc4 and ynew of STEP3)
% 
%                   The current quantized stepsizes are used to compute the new error estimate 
%                   "err_est" 
% 
%               If err_est > TOL
%                   {
%                      failed attempt:
%                      calculate smaller step size stsize = 0.7* h
%                      and return to  SUBS 2.A.1 STEP1 
%                    }
% 
%               }
%               (end of SUBLOOP Repeat the following three steps until err_est <= TOL)
% 
% 
%               The current step is accepted:
% 
%            SUBS 2.B STEPSIZE SELECTION:
% 
%               The next three consecutive stepsizes are obtained by the following formula 
%                    stsize  = h *stfac *(TOL/err_est )^(1.0/14.0) ;  
%               "stsize" are then quantized and stored back into "stsize"
% 
%            SUBS 2.C INTERPOLATION  
%                if  (tc > tend) use interpolation procedure "id123O14" to get ynew
% 
%           }
%          (end of MAIN LOOP  to advance integration from x_n to x_{n+3} per 1 iteration)
% 
% It is to be noted that the section number and name mentioned above are identical to  
% the section number and name appeared in the code 


*/

	int nbcmpP1=nbcmp+1;
	// p 140 /John Hubbard: arrays
	double Qbv1_7[8],Qbv8_11[5], Qbv12_15[5], lra, rra  ;
	double bv1_7[8], bv8_11[5], bv12_15[5];
	
	int idx2[20], ilr, irr,  idx2ilr, idx2irr ;
	int m14jd, m13jd,m12jd,m11jd,m10jd,m9jd,m8jd,m7jd ;
	double xtab[20];
	double xkm14, xkm13, xkm12, xkm11, xkm10,  xkm9 , xkm8 , xkm7 , xkm6 , xkm5 , xkm4 , xkm3 , xkm2 ;
	double xkm1 , xk1, xk2,xk3,xk4,xk5,xk6, c1, c2,  c3, c4, c5, c6, c7  ;
    double preh, nstsize, xk1mxkm1 ;	
	double he2, he3;
	double yc[nbcmpP1];	
    int j,  nbinitst ;

	// "comparing.."Hull, Enright,..p624 the true sol is computed by a 5th order RK with tol=1% of TOL
	
	//------- PARAM: -------
	
	double hmax = 0.4*fabs(tend-t0)  ;
	double stfac=0.58 ;
//	double stfac=0.58 ;
//	double stfac=0.64 ;
//	double stfac=0.71 ;
//	double stfac=0.81 ;
	double hv, hm1 ;	
	nbinitst = 4;
	int nbeps=6 ;
//	int iniordk=4 ;
	
	//------- END PARAM ----

	//Output file
	// FILE* fid=fopen("outp.txt","a");
	
	//prgt0=t0 prgy0=y0
	
	//  eps of hmin:  eps = 2.220446049250313e-16
	double eps = 2.220446049250313e-16 ;
		
	int q= 1;
	
	int nbpos = 10;
	

// ***** 16 comb. HBO144 coeffs  : ******************

// % ************ SECTION 1. INITIALISATION **************************

//  *****  SUBS 1.A  INPUT THE MATRIX OF COEFFICIENTS: HBO144cfs.cpp ***
#include "HBO144cfs.cpp"//HBO144 coeffs 
	
	double tc = t0;

	for(int i=1;i<=nbcmp;i++)
		yc[i]=y0[i];
		
	int k=1111;
	j=0;
	int oc=3 ;
	
	// 3 HBO4 init. steps :
	double resJandH[2];


// ********* SUBS 1.B THE FIRST FOUR EQUAL CONSECUTIVE STEPSIZES
//              are obtained by the procedure "HBO7_4initd123steps"       

	HBO7inicnstps(RELTOL, TOL, t0, tend, y0, nbinitst, tvals, yvals, fvals, res,f2vals,f3vals, resJandH,nbcmp);
	j=(int)resJandH[0];
	double h=resJandH[1];
	
	hv=h;
		
	int temp=j+nbpos;
	for(int i=1;i<=nbcmp;i++)
		yc[i]=yvals[j+nbpos][i];
		
	tc = tvals[temp];
		
	//%% (((INIT tvals t_{-1}, t_{-2} ...: OK
	for( int idx=4;idx<=12;idx++){
		tvals[temp-idx]=tc-idx*h ;
	}
	//%% INIT tvals t_{-1}, t_{-2} ... )))
				
	double MAXGE = -9999.0;
	oc=1111 ;
	int abmhbnfe=nbinitst*2 ;		
	//
		
	int nreje=0;
	double hnew = h ;
		
// INIT PARAM.
// RK4 coeffs:

 c1 = 0 ;  c2 =     1.0/3.0 ;
 c3 =     2.0/3.0   ; c4 =     1.0  ;
			
	//  of hmin:  compeps = 2.220446049250313e-16
	double compeps = 2.220446049250313e-16 ;
	double hmin = 16*compeps*fabs(tc);
		
	//     -->    col1_2(:,1)=  yvals(m0jd ,:); ...
	double col1_7[8][nbcmpP1];
	double col8_11[5][nbcmpP1];	
	double col12_15[5][nbcmpP1];	

	double erk= 0;
	int nbrep=0;	

	double yk1[nbcmpP1];
	double ynew[nbcmpP1];
	double y_est[nbcmpP1];

	double y_jpc2[nbcmpP1];
	double y_jpc3[nbcmpP1];
	double y_jpc4[nbcmpP1];
			
	double fnewk2[nbcmpP1];
	double fnewk3[nbcmpP1];
	double fnewk4[nbcmpP1];	

	double**fnewRes=new double*[3];
	fnewRes[0]=new double[nbcmpP1];
	fnewRes[1]=new double[nbcmpP1];
	fnewRes[2]=new double[nbcmpP1];
				
	double b_v[20];

	double err_estkp1;
	double err_estkm1;
	double err_estkm2;
	double hold;
	double tnew;
	double stsize;
	int jpnbposp1;
	double prevh;
	double err_est;
	int jpnbpos;

// **** SECTION 2. PERFORM THE INTEGRATION : 
//      MAIN LOOP TO ADVANCE INTEGRATION FROM x_n to x_{n+3} PER ITERATION *****************

	bool Nret=true;
	while(  tvals[j+nbpos] < tend && Nret){
				
		hmin = 16*eps*fabs(tc);
					
		if( j >= nbinitst+1000000 ){
    	    cout << " HBO144q16: j= " << j << ", TOL =" << TOL << endl ;

            hm1= tvals[j+nbpos]-tvals[j-1+nbpos]  ;
	        cout << "  j= " << j << ", hm8 =" << tvals[j-7+nbpos]-tvals[j-8+nbpos]  << endl ;
	        cout << "  j= " << j << ", hm7 =" << tvals[j-6+nbpos]-tvals[j-7+nbpos]  << endl ;
	        cout << "  j= " << j << ", hm6 =" << tvals[j-5+nbpos]-tvals[j-6+nbpos]  << endl ;
	        cout << "  j= " << j << ", hm5 =" << tvals[j-4+nbpos]-tvals[j-5+nbpos]  << endl ;
	        cout << "  j= " << j << ", hm4 =" << tvals[j-3+nbpos]-tvals[j-4+nbpos]  << endl ;
	        cout << "  j= " << j << ", hm3 =" << tvals[j-2+nbpos]-tvals[j-3+nbpos]  << endl ;
	        cout << "  j= " << j << ", hm2 =" << tvals[j-1+nbpos]-tvals[j-2+nbpos]  << endl ;
	        cout << "  j= " << j << ", hm1 =" << tvals[j+nbpos]-tvals[j-1+nbpos]  << endl ;
	        cout << "  tvals= " << tvals[j+nbpos ]  << endl ;
			
	        cout << "  yvalszero= " << yvals[j+nbpos ][1]-exp(-tvals[j+nbpos ]) << endl ;
									
			cout<< "Debug stop." << endl;
//			int err;
//			cin>>err;
			Nret=false;
            exit(1);
		}
		
		h=hnew;				
		// ------- (MAIN INTEG STEP) --------------						
		err_est=9999;		
		jpnbpos=j+nbpos;
									
		int m6jd=jpnbpos-6  ,m5jd=jpnbpos-5  ,m4jd=jpnbpos-4  ,m3jd=jpnbpos-3  ;
		int m2jd=jpnbpos-2  ,m1jd=jpnbpos-1  ,m0jd=jpnbpos  ;

		preh=tvals[jpnbpos]-tvals[jpnbpos-1] ;
		

		// (SUBS 2.A) SUBLOOP REPEAT THE FOLLOWING FOUR STEPS UNTIL err_est <= TOL 
		//               to advance integration from x_n to x_{n+3}

		while ( err_est > TOL  ){									
		
            // **** SUBS 2.A.1 STEP1 : TO ADVANCE INTEGRATION FROM x_n to x_{n+1} ****
		
			xkm14 =0.0 ;
			xkm13 = 1.0e-11  ;
			xkm12 = 2.0e-11  ;
			xkm11 = 3.0e-11  ;
			xkm10 = 4.0e-11  ;
			xkm9 =  5.0e-11  ;
			xkm8 =  6.0e-11  ;
			xkm7 =  7.0e-11  ;
			xkm6 = tvals[m6jd] ;
			xkm5 = tvals[m5jd] ;
			xkm4 = tvals[m4jd];
			xkm3 = tvals[m3jd] ;			
			xkm2 = tvals[m2jd] ;
			xkm1 = tvals[m1jd] ;
			xk1 =  tvals[m0jd];
			tc=xk1 ;			
			xk2 = tvals[jpnbpos] + c2* h;
			xk3 = tvals[jpnbpos] + c3*h,       
			xk4 = tvals[jpnbpos] + c4*h;						
   xk1mxkm1 = xk1-xkm1 ;
   rra=h/xk1mxkm1  ;	
   
//   ilr=int(lra*10+0.5) ;
   irr=int(rra*10+0.5) ;   

    // ************  PART : Integration Formula coeffs ***********
   
    Qbv1_7[1]     =cfos4_1[ irr]  ;
    Qbv1_7[2]     =cfos4_2[ irr]  ;
    Qbv1_7[3]     =cfos4_3[ irr]  ;
    Qbv1_7[4]     =cfos4_4[ irr]  ;
    Qbv1_7[5]     =cfos4_5[ irr]  ;		
    Qbv1_7[6]     =cfos4_6[ irr]  ;
    Qbv1_7[7]     =cfos4_7[ irr]  ;
		
    Qbv8_11[1]   =cfos4_8[ irr]  ;
    Qbv8_11[2]   =cfos4_9[ irr]  ;
    Qbv8_11[3]   =cfos4_10[ irr]  ;
    Qbv8_11[4]   =cfos4_11[ irr]  ;

    Qbv12_15[1] =cfos4_12[ irr]  ;
    Qbv12_15[2] =cfos4_13[ irr]  ;
    Qbv12_15[3] =cfos4_14[ irr]  ;
    Qbv12_15[4] =cfos4_15[ irr]  ;
					
	for(int i=1;i<=nbcmp;i++){
		yk1[i]= yvals[m0jd][i] ;
				
		col1_7[4][i] =fvals[ m0jd][i];
		col1_7[5][i] =fvals[ m1jd][i];
		col1_7[6][i] =fvals[ m2jd][i];				
		col1_7[7][i] =fvals[ m3jd][i];						
								
		col8_11[1][i] =f2vals[ m0jd][i];
		col8_11[2][i] =f2vals[ m1jd][i];
		col8_11[3][i] =f2vals[ m2jd][i];
		col8_11[4][i] =f2vals[ m3jd][i];				
						
		col12_15[1][i] =f3vals[ m0jd][i];
		col12_15[2][i] =f3vals[ m1jd][i];
		col12_15[3][i] =f3vals[ m2jd][i];
		col12_15[4][i] =f3vals[ m3jd][i];
		}
       // end  for(int i=1;i<=nbcmp;i++)
	   
   //   *********** PART : predictor P2   ***********
   //		(to obtain the value y_{n+c2})	   

	bv1_7[1]  =0.0  ;
	bv1_7[2]  =0.0  ;
	bv1_7[3]  =0.0  ;		
	bv1_7[4]  =cfos1_1[irr]  ;
    bv1_7[5]  =cfos1_2[irr]  ;
    bv1_7[6]  =cfos1_3[irr]  ;
    bv1_7[7]  =cfos1_4[irr]  ;
	
    bv8_11[1]  =cfos1_5[irr]  ;
    bv8_11[2]  =cfos1_6[irr]  ;
    bv8_11[3]  =cfos1_7[irr]  ;
    bv8_11[4]  =cfos1_8[irr]  ;

    bv12_15[1]  =cfos1_9[irr]  ;
    bv12_15[2]  =cfos1_10[irr]  ;
    bv12_15[3]  =cfos1_11[irr]  ;
    bv12_15[4]  =cfos1_12[irr]  ;	
	
    he2=h*h ;
    he3=he2*h ;
	
	for(int i=1;i<=nbcmp;i++){
		y_jpc2[i] = yk1[i] + h*( col1_7[4][i]*bv1_7[4]+ col1_7[5][i]*bv1_7[5]+ col1_7[6][i]*bv1_7[6]+ col1_7[7][i]*bv1_7[7]) + he2*( col8_11[1][i]*bv8_11[1]+ col8_11[2][i]*bv8_11[2]+ col8_11[3][i]*bv8_11[3]+ col8_11[4][i]*bv8_11[4]) + he3*( col12_15[1][i]*bv12_15[1]+ col12_15[2][i]*bv12_15[2]+ col12_15[3][i]*bv12_15[3]+ col12_15[4][i]*bv12_15[4]) ;
	    }				
	f(xk2  , y_jpc2,nbcmp,fnewk2);
	for(int i=1;i<=nbcmp;i++)
		col1_7[3][i]=fnewk2[i] ;

    //  ************  PART : predictor P3  ***********
    //		(to obtain the value y_{n+c3})
			
	bv1_7[1]  =0.0  ;
	bv1_7[2]  =0.0  ;			
	bv1_7[3]  =cfos2_1[irr]  ;
    bv1_7[4]  =cfos2_2[irr]  ;
    bv1_7[5]  =cfos2_3[irr]  ;
    bv1_7[6]  =cfos2_4[irr]  ;
    bv1_7[7]  =cfos2_5[irr]  ;
		
    bv8_11[1]  =cfos2_6[irr]  ;
    bv8_11[2]  =cfos2_7[irr]  ;
    bv8_11[3]  =cfos2_8[irr]  ;
    bv8_11[4]  =cfos2_9[irr]  ;

    bv12_15[1]  =cfos2_10[irr]  ;
    bv12_15[2]  =cfos2_11[irr]  ;
    bv12_15[3]  =cfos2_12[irr]  ;
    bv12_15[4]  =cfos2_13[irr]  ;		
		
	for(int i=1;i<=nbcmp;i++){
		y_jpc3[i] = yk1[i] + h*( col1_7[3][i]*bv1_7[3]+col1_7[4][i]*bv1_7[4]+ col1_7[5][i]*bv1_7[5]+ col1_7[6][i]*bv1_7[6]+ col1_7[7][i]*bv1_7[7]) + he2*( col8_11[1][i]*bv8_11[1]+ col8_11[2][i]*bv8_11[2]+ col8_11[3][i]*bv8_11[3]+ col8_11[4][i]*bv8_11[4]) + he3*( col12_15[1][i]*bv12_15[1]+ col12_15[2][i]*bv12_15[2]+ col12_15[3][i]*bv12_15[3]+ col12_15[4][i]*bv12_15[4]) ;
	    }
		   			
	f(xk3  , y_jpc3,nbcmp,fnewk3);
	for(int i=1;i<=nbcmp;i++)
		col1_7[2][i]=fnewk3[i] ;

    //  ************  PART : predictor P4  ***********
    //		(to obtain the value y_{n+c4})

	bv1_7[1]  =0.0  ;			
	bv1_7[2]  =cfos3_1[irr]  ;
    bv1_7[3]  =cfos3_2[irr]  ;
    bv1_7[4]  =cfos3_3[irr]  ;
    bv1_7[5]  =cfos3_4[irr]  ;
    bv1_7[6]  =cfos3_5[irr]  ;
    bv1_7[7]  =cfos3_6[irr]  ;
		
    bv8_11[1]  =cfos3_7[irr]  ;
    bv8_11[2]  =cfos3_8[irr]  ;
    bv8_11[3]  =cfos3_9[irr]  ;
    bv8_11[4]  =cfos3_10[irr]  ;

    bv12_15[1]  =cfos3_11[irr]  ;
    bv12_15[2]  =cfos3_12[irr]  ;
    bv12_15[3]  =cfos3_13[irr]  ;
    bv12_15[4]  =cfos3_14[irr]  ;		
				
	for(int i=1;i<=nbcmp;i++){
		y_jpc4[i] = yk1[i] + h*( col1_7[2][i]*bv1_7[2]+col1_7[3][i]*bv1_7[3]+ col1_7[4][i]*bv1_7[4]+ col1_7[5][i]*bv1_7[5] +col1_7[6][i]*bv1_7[6] +col1_7[7][i]*bv1_7[7] ) + he2*( col8_11[1][i]*bv8_11[1]+ col8_11[2][i]*bv8_11[2]+ col8_11[3][i]*bv8_11[3]+ col8_11[4][i]*bv8_11[4]) + he3*( col12_15[1][i]*bv12_15[1]+ col12_15[2][i]*bv12_15[2]+ col12_15[3][i]*bv12_15[3]+ col12_15[4][i]*bv12_15[4]) ;
	    }

	f(xk4  , y_jpc4,nbcmp,fnewk4);
	for(int i=1;i<=nbcmp;i++)
		col1_7[1][i]=fnewk4[i] ;

    //  ************  PART : Integration Formula ***********
    //		(to obtain the value y_{n+1})
		
	for(int i=1;i<=nbcmp;i++){
		ynew[i] = yk1[i] + h*( col1_7[1][i]*Qbv1_7[1]+col1_7[2][i]*Qbv1_7[2]+col1_7[3][i]*Qbv1_7[3]+ col1_7[4][i]*Qbv1_7[4]+ col1_7[5][i]*Qbv1_7[5] +col1_7[6][i]*Qbv1_7[6]+col1_7[7][i]*Qbv1_7[7]  ) + he2*( col8_11[1][i]*Qbv8_11[1]+ col8_11[2][i]*Qbv8_11[2]+ col8_11[3][i]*Qbv8_11[3]+ col8_11[4][i]*Qbv8_11[4]) + he3*( col12_15[1][i]*Qbv12_15[1]+ col12_15[2][i]*Qbv12_15[2]+ col12_15[3][i]*Qbv12_15[3]+ col12_15[4][i]*Qbv12_15[4]) ;
	    }		
		tnew = tc + h;
		fpp(tnew, ynew,nbcmp,fnewRes) ;		
		
// ************ STEP1 )) ************

		jpnbposp1=j+nbpos+1;			
		for(int i=1;i<=nbcmp;i++)	{
			yvals[jpnbposp1][i] = ynew[i] ;
			fvals[jpnbposp1][i]=  fnewRes[0][i] ;
			f2vals[jpnbposp1][i]=  fnewRes[1][i] ;
			f3vals[jpnbposp1][i]=  fnewRes[2][i] ;			
		}
		tvals[jpnbposp1] =  tnew;
        j =j+1 ;


//  ***** SUBS 2.A.2 STEP2 : TO ADVANCE INTEGRATION FROM x_{n+1} to x_{n+2} ****

		jpnbpos=j+nbpos;
									
		int m6jd=jpnbpos-6  ,m5jd=jpnbpos-5  ,m4jd=jpnbpos-4  ,m3jd=jpnbpos-3  ;
		int m2jd=jpnbpos-2  ,m1jd=jpnbpos-1  ,m0jd=jpnbpos  ;

			xkm6 = tvals[m6jd] ;
			xkm5 = tvals[m5jd] ;
			xkm4 = tvals[m4jd];
			xkm3 = tvals[m3jd] ;			
			xkm2 = tvals[m2jd] ;
			xkm1 = tvals[m1jd] ;
			xk1 =  tvals[m0jd];
			tc=xk1 ;			
			xk2 = tvals[jpnbpos] + c2* h;
			xk3 = tvals[jpnbpos] + c3*h,       
			xk4 = tvals[jpnbpos] + c4*h;

    //  ************  PART : Integration Formula ***********

    Qbv1_7[1]     =cfos8_1[ irr]  ;
    Qbv1_7[2]     =cfos8_2[ irr]  ;
    Qbv1_7[3]     =cfos8_3[ irr]  ;
    Qbv1_7[4]     =cfos8_4[ irr]  ;
    Qbv1_7[5]     =cfos8_5[ irr]  ;		
    Qbv1_7[6]     =cfos8_6[ irr]  ;
    Qbv1_7[7]     =cfos8_7[ irr]  ;
		
    Qbv8_11[1]   =cfos8_8[ irr]  ;
    Qbv8_11[2]   =cfos8_9[ irr]  ;
    Qbv8_11[3]   =cfos8_10[ irr]  ;
    Qbv8_11[4]   =cfos8_11[ irr]  ;

    Qbv12_15[1] =cfos8_12[ irr]  ;
    Qbv12_15[2] =cfos8_13[ irr]  ;
    Qbv12_15[3] =cfos8_14[ irr]  ;
    Qbv12_15[4] =cfos8_15[ irr]  ;

	for(int i=1;i<=nbcmp;i++){
		yk1[i]= yvals[m0jd][i] ;
				
		col1_7[4][i] =fvals[ m0jd][i];
		col1_7[5][i] =fvals[ m1jd][i];
		col1_7[6][i] =fvals[ m2jd][i];				
		col1_7[7][i] =fvals[ m3jd][i];						
								
		col8_11[1][i] =f2vals[ m0jd][i];
		col8_11[2][i] =f2vals[ m1jd][i];
		col8_11[3][i] =f2vals[ m2jd][i];
		col8_11[4][i] =f2vals[ m3jd][i];				
						
		col12_15[1][i] =f3vals[ m0jd][i];
		col12_15[2][i] =f3vals[ m1jd][i];
		col12_15[3][i] =f3vals[ m2jd][i];
		col12_15[4][i] =f3vals[ m3jd][i];
		}

		
   // ************  PART : predictor P2  ***********
   //		(to obtain the value y_{n+1+c2})

	bv1_7[1]  =0.0  ;
	bv1_7[2]  =0.0  ;
	bv1_7[3]  =0.0  ;		
	bv1_7[4]  =cfos5_1[irr]  ;
    bv1_7[5]  =cfos5_2[irr]  ;
    bv1_7[6]  =cfos5_3[irr]  ;
    bv1_7[7]  =cfos5_4[irr]  ;
	
    bv8_11[1]  =cfos5_5[irr]  ;
    bv8_11[2]  =cfos5_6[irr]  ;
    bv8_11[3]  =cfos5_7[irr]  ;
    bv8_11[4]  =cfos5_8[irr]  ;

    bv12_15[1]  =cfos5_9[irr]  ;
    bv12_15[2]  =cfos5_10[irr]  ;
    bv12_15[3]  =cfos5_11[irr]  ;
    bv12_15[4]  =cfos5_12[irr]  ;	
	
	for(int i=1;i<=nbcmp;i++){
		y_jpc2[i] = yk1[i] + h*( col1_7[4][i]*bv1_7[4]+ col1_7[5][i]*bv1_7[5]+ col1_7[6][i]*bv1_7[6]+ col1_7[7][i]*bv1_7[7]) + he2*( col8_11[1][i]*bv8_11[1]+ col8_11[2][i]*bv8_11[2]+ col8_11[3][i]*bv8_11[3]+ col8_11[4][i]*bv8_11[4]) + he3*( col12_15[1][i]*bv12_15[1]+ col12_15[2][i]*bv12_15[2]+ col12_15[3][i]*bv12_15[3]+ col12_15[4][i]*bv12_15[4]) ;
	    }
	
	f(xk2  , y_jpc2,nbcmp,fnewk2);
	for(int i=1;i<=nbcmp;i++)
		col1_7[3][i]=fnewk2[i] ;

   //   ************  PART : predictor P3  ***********
   //		(to obtain the value y_{n+1+c3})

	bv1_7[1]  =0.0  ;
	bv1_7[2]  =0.0  ;			
	bv1_7[3]  =cfos6_1[irr]  ;
    bv1_7[4]  =cfos6_2[irr]  ;
    bv1_7[5]  =cfos6_3[irr]  ;
    bv1_7[6]  =cfos6_4[irr]  ;
    bv1_7[7]  =cfos6_5[irr]  ;
		
    bv8_11[1]  =cfos6_6[irr]  ;
    bv8_11[2]  =cfos6_7[irr]  ;
    bv8_11[3]  =cfos6_8[irr]  ;
    bv8_11[4]  =cfos6_9[irr]  ;

    bv12_15[1]  =cfos6_10[irr]  ;
    bv12_15[2]  =cfos6_11[irr]  ;
    bv12_15[3]  =cfos6_12[irr]  ;
    bv12_15[4]  =cfos6_13[irr]  ;		
	
	for(int i=1;i<=nbcmp;i++){
		y_jpc3[i] = yk1[i] + h*( col1_7[3][i]*bv1_7[3]+col1_7[4][i]*bv1_7[4]+ col1_7[5][i]*bv1_7[5]+ col1_7[6][i]*bv1_7[6]+ col1_7[7][i]*bv1_7[7]) + he2*( col8_11[1][i]*bv8_11[1]+ col8_11[2][i]*bv8_11[2]+ col8_11[3][i]*bv8_11[3]+ col8_11[4][i]*bv8_11[4]) + he3*( col12_15[1][i]*bv12_15[1]+ col12_15[2][i]*bv12_15[2]+ col12_15[3][i]*bv12_15[3]+ col12_15[4][i]*bv12_15[4]) ;
	    }
					   			
	f(xk3  , y_jpc3,nbcmp,fnewk3);
	for(int i=1;i<=nbcmp;i++)
		col1_7[2][i]=fnewk3[i] ;

   //   ************  PART : predictor P4  ***********
   //		(to obtain the value y_{n+1+c4})

	bv1_7[1]  =0.0  ;			
	bv1_7[2]  =cfos7_1[irr]  ;
    bv1_7[3]  =cfos7_2[irr]  ;
    bv1_7[4]  =cfos7_3[irr]  ;
    bv1_7[5]  =cfos7_4[irr]  ;
    bv1_7[6]  =cfos7_5[irr]  ;
    bv1_7[7]  =cfos7_6[irr]  ;
		
    bv8_11[1]  =cfos7_7[irr]  ;
    bv8_11[2]  =cfos7_8[irr]  ;
    bv8_11[3]  =cfos7_9[irr]  ;
    bv8_11[4]  =cfos7_10[irr]  ;

    bv12_15[1]  =cfos7_11[irr]  ;
    bv12_15[2]  =cfos7_12[irr]  ;
    bv12_15[3]  =cfos7_13[irr]  ;
    bv12_15[4]  =cfos7_14[irr]  ;		
			
	for(int i=1;i<=nbcmp;i++){
		y_jpc4[i] = yk1[i] + h*( col1_7[2][i]*bv1_7[2]+col1_7[3][i]*bv1_7[3]+ col1_7[4][i]*bv1_7[4]+ col1_7[5][i]*bv1_7[5] +col1_7[6][i]*bv1_7[6] +col1_7[7][i]*bv1_7[7] ) + he2*( col8_11[1][i]*bv8_11[1]+ col8_11[2][i]*bv8_11[2]+ col8_11[3][i]*bv8_11[3]+ col8_11[4][i]*bv8_11[4]) + he3*( col12_15[1][i]*bv12_15[1]+ col12_15[2][i]*bv12_15[2]+ col12_15[3][i]*bv12_15[3]+ col12_15[4][i]*bv12_15[4]) ;
	    }

	f(xk4  , y_jpc4,nbcmp,fnewk4);
	for(int i=1;i<=nbcmp;i++)
		col1_7[1][i]=fnewk4[i] ;


    //  ************  PART : Integration Formula   ***********
    //		(to obtain the value y_{n+2})
				
	for(int i=1;i<=nbcmp;i++){
		ynew[i] = yk1[i] + h*( col1_7[1][i]*Qbv1_7[1]+col1_7[2][i]*Qbv1_7[2]+col1_7[3][i]*Qbv1_7[3]+ col1_7[4][i]*Qbv1_7[4]+ col1_7[5][i]*Qbv1_7[5] +col1_7[6][i]*Qbv1_7[6] +col1_7[7][i]*Qbv1_7[7]  ) + he2*( col8_11[1][i]*Qbv8_11[1]+ col8_11[2][i]*Qbv8_11[2]+ col8_11[3][i]*Qbv8_11[3]+ col8_11[4][i]*Qbv8_11[4]) + he3*( col12_15[1][i]*Qbv12_15[1]+ col12_15[2][i]*Qbv12_15[2]+ col12_15[3][i]*Qbv12_15[3]+ col12_15[4][i]*Qbv12_15[4]) ;
	    }

		tnew = tc + h;
		fpp(tnew, ynew,nbcmp,fnewRes) ;
		
// ************  STEP2 )) ******************************************		

		jpnbposp1=j+nbpos+1;			
		for(int i=1;i<=nbcmp;i++)	{
			yvals[jpnbposp1][i] = ynew[i] ;
			fvals[jpnbposp1][i]=  fnewRes[0][i] ;
			f2vals[jpnbposp1][i]=  fnewRes[1][i] ;
			f3vals[jpnbposp1][i]=  fnewRes[2][i] ;			
		}
		tvals[jpnbposp1] =  tnew;
        j =j+1 ;


// ***** SUBS 2.A.3 STEP3 : TO ADVANCE INTEGRATION FROM x_{n+2} to x_{n+3} ****

		jpnbpos=j+nbpos;									
		m6jd=jpnbpos-6  ,m5jd=jpnbpos-5  ,m4jd=jpnbpos-4  ,m3jd=jpnbpos-3  ;
		m2jd=jpnbpos-2  ,m1jd=jpnbpos-1  ,m0jd=jpnbpos  ;

			xkm6 = tvals[m6jd] ;
			xkm5 = tvals[m5jd] ;
			xkm4 = tvals[m4jd];
			xkm3 = tvals[m3jd] ;			
			xkm2 = tvals[m2jd] ;
			xkm1 = tvals[m1jd] ;
			xk1 =  tvals[m0jd];
			tc=xk1 ;			
			xk2 = tvals[jpnbpos] + c2* h;
			xk3 = tvals[jpnbpos] + c3*h,       
			xk4 = tvals[jpnbpos] + c4*h;

    Qbv1_7[1]     =cfos12_1[ irr]  ;
    Qbv1_7[2]     =cfos12_2[ irr]  ;
    Qbv1_7[3]     =cfos12_3[ irr]  ;
    Qbv1_7[4]     =cfos12_4[ irr]  ;
    Qbv1_7[5]     =cfos12_5[ irr]  ;		
    Qbv1_7[6]     =cfos12_6[ irr]  ;
    Qbv1_7[7]     =cfos12_7[ irr]  ;
		
    Qbv8_11[1]   =cfos12_8[ irr]  ;
    Qbv8_11[2]   =cfos12_9[ irr]  ;
    Qbv8_11[3]   =cfos12_10[ irr]  ;
    Qbv8_11[4]   =cfos12_11[ irr]  ;

    Qbv12_15[1] =cfos12_12[ irr]  ;
    Qbv12_15[2] =cfos12_13[ irr]  ;
    Qbv12_15[3] =cfos12_14[ irr]  ;
    Qbv12_15[4] =cfos12_15[ irr]  ;

	for(int i=1;i<=nbcmp;i++){
		yk1[i]= yvals[m0jd][i] ;
				
		col1_7[4][i] =fvals[ m0jd][i];
		col1_7[5][i] =fvals[ m1jd][i];
		col1_7[6][i] =fvals[ m2jd][i];				
		col1_7[7][i] =fvals[ m3jd][i];						
								
		col8_11[1][i] =f2vals[ m0jd][i];
		col8_11[2][i] =f2vals[ m1jd][i];
		col8_11[3][i] =f2vals[ m2jd][i];
		col8_11[4][i] =f2vals[ m3jd][i];				
						
		col12_15[1][i] =f3vals[ m0jd][i];
		col12_15[2][i] =f3vals[ m1jd][i];
		col12_15[3][i] =f3vals[ m2jd][i];
		col12_15[4][i] =f3vals[ m3jd][i];
		}
		
   // ************  PART : predictor P2  ***********
   //		(to obtain the value y_{n+2+c2})
		
	bv1_7[1]  =0.0  ;
	bv1_7[2]  =0.0  ;
	bv1_7[3]  =0.0  ;		
	bv1_7[4]  =cfos9_1[irr]  ;
    bv1_7[5]  =cfos9_2[irr]  ;
    bv1_7[6]  =cfos9_3[irr]  ;
    bv1_7[7]  =cfos9_4[irr]  ;
	
    bv8_11[1]  =cfos9_5[irr]  ;
    bv8_11[2]  =cfos9_6[irr]  ;
    bv8_11[3]  =cfos9_7[irr]  ;
    bv8_11[4]  =cfos9_8[irr]  ;

    bv12_15[1]  =cfos9_9[irr]  ;
    bv12_15[2]  =cfos9_10[irr]  ;
    bv12_15[3]  =cfos9_11[irr]  ;
    bv12_15[4]  =cfos9_12[irr]  ;	
	
	for(int i=1;i<=nbcmp;i++){
		y_jpc2[i] = yk1[i] + h*( col1_7[4][i]*bv1_7[4]+ col1_7[5][i]*bv1_7[5]+ col1_7[6][i]*bv1_7[6]+ col1_7[7][i]*bv1_7[7]) + he2*( col8_11[1][i]*bv8_11[1]+ col8_11[2][i]*bv8_11[2]+ col8_11[3][i]*bv8_11[3]+ col8_11[4][i]*bv8_11[4]) + he3*( col12_15[1][i]*bv12_15[1]+ col12_15[2][i]*bv12_15[2]+ col12_15[3][i]*bv12_15[3]+ col12_15[4][i]*bv12_15[4]) ;
	    }

	f(xk2  , y_jpc2,nbcmp,fnewk2);
	for(int i=1;i<=nbcmp;i++)
		col1_7[3][i]=fnewk2[i] ;

   // ************  PART : predictor P3  ***********
   //		(to obtain the value y_{n+2+c3})

	bv1_7[1]  =0.0  ;
	bv1_7[2]  =0.0  ;			
	bv1_7[3]  =cfos10_1[irr]  ;
    bv1_7[4]  =cfos10_2[irr]  ;
    bv1_7[5]  =cfos10_3[irr]  ;
    bv1_7[6]  =cfos10_4[irr]  ;
    bv1_7[7]  =cfos10_5[irr]  ;
		
    bv8_11[1]  =cfos10_6[irr]  ;
    bv8_11[2]  =cfos10_7[irr]  ;
    bv8_11[3]  =cfos10_8[irr]  ;
    bv8_11[4]  =cfos10_9[irr]  ;

    bv12_15[1]  =cfos10_10[irr]  ;
    bv12_15[2]  =cfos10_11[irr]  ;
    bv12_15[3]  =cfos10_12[irr]  ;
    bv12_15[4]  =cfos10_13[irr]  ;	

	for(int i=1;i<=nbcmp;i++){
		y_jpc3[i] = yk1[i] + h*( col1_7[3][i]*bv1_7[3]+col1_7[4][i]*bv1_7[4]+ col1_7[5][i]*bv1_7[5]+ col1_7[6][i]*bv1_7[6]+ col1_7[7][i]*bv1_7[7]) + he2*( col8_11[1][i]*bv8_11[1]+ col8_11[2][i]*bv8_11[2]+ col8_11[3][i]*bv8_11[3]+ col8_11[4][i]*bv8_11[4]) + he3*( col12_15[1][i]*bv12_15[1]+ col12_15[2][i]*bv12_15[2]+ col12_15[3][i]*bv12_15[3]+ col12_15[4][i]*bv12_15[4]) ;
	    }
	   			
	f(xk3  , y_jpc3,nbcmp,fnewk3);
	for(int i=1;i<=nbcmp;i++)
		col1_7[2][i]=fnewk3[i] ;

   // ************  PART : predictor P4  ***********
   //		(to obtain the value y_{n+2+c4})

	bv1_7[1]  =0.0  ;			
	bv1_7[2]  =cfos11_1[irr]  ;
    bv1_7[3]  =cfos11_2[irr]  ;
    bv1_7[4]  =cfos11_3[irr]  ;
    bv1_7[5]  =cfos11_4[irr]  ;
    bv1_7[6]  =cfos11_5[irr]  ;
    bv1_7[7]  =cfos11_6[irr]  ;
		
    bv8_11[1]  =cfos11_7[irr]  ;
    bv8_11[2]  =cfos11_8[irr]  ;
    bv8_11[3]  =cfos11_9[irr]  ;
    bv8_11[4]  =cfos11_10[irr]  ;

    bv12_15[1]  =cfos11_11[irr]  ;
    bv12_15[2]  =cfos11_12[irr]  ;
    bv12_15[3]  =cfos11_13[irr]  ;
    bv12_15[4]  =cfos11_14[irr]  ;		
		
	for(int i=1;i<=nbcmp;i++){
		y_jpc4[i] = yk1[i] + h*( col1_7[2][i]*bv1_7[2]+col1_7[3][i]*bv1_7[3]+ col1_7[4][i]*bv1_7[4]+ col1_7[5][i]*bv1_7[5] +col1_7[6][i]*bv1_7[6] +col1_7[7][i]*bv1_7[7] ) + he2*( col8_11[1][i]*bv8_11[1]+ col8_11[2][i]*bv8_11[2]+ col8_11[3][i]*bv8_11[3]+ col8_11[4][i]*bv8_11[4]) + he3*( col12_15[1][i]*bv12_15[1]+ col12_15[2][i]*bv12_15[2]+ col12_15[3][i]*bv12_15[3]+ col12_15[4][i]*bv12_15[4]) ;
	    }

	f(xk4  , y_jpc4,nbcmp,fnewk4);
	for(int i=1;i<=nbcmp;i++)
		col1_7[1][i]=fnewk4[i] ;
	
//	exit(1) ;	
// %%%%%%%%%%%%%%%%%%%%%  STEP3 )) %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%		


    //  ************  PART : Integration Formula  ***********
    //       (to obtain the value y_{n+3})
    //            and Step Control Predictor  
    //       (to obtain the value \widetilde{y}_{n+3})
    //
    //   The current quantized stepsizes are used to compute the new error estimate 
    //                "err_est" 
			
	err_est = 0.0 ;

    bv1_7[1]     =cfos13_1[ irr]  ;
    bv1_7[2]     =cfos13_2[ irr]  ;
    bv1_7[3]     =cfos13_3[ irr]  ;
    bv1_7[4]     =cfos13_4[ irr]  ;
    bv1_7[5]     =cfos13_5[ irr]  ;		
    bv1_7[6]     =cfos13_6[ irr]  ;
    bv1_7[7]     =cfos13_7[ irr]  ;
		
    bv8_11[1]   =cfos13_8[ irr]  ;
    bv8_11[2]   =cfos13_9[ irr]  ;
    bv8_11[3]   =cfos13_10[ irr]  ;
    bv8_11[4]   =cfos13_11[ irr]  ;

    bv12_15[1] =cfos13_12[ irr]  ;
    bv12_15[2] =cfos13_13[ irr]  ;
    bv12_15[3] =cfos13_14[ irr]  ;
    bv12_15[4] =cfos13_15[ irr]  ;
		
//// note: col8_11[5][i]*0 = col8_11[5][i]*b2(b2=0)
	for(int i=1;i<=nbcmp;i++){
		ynew[i] = yk1[i] + h*( col1_7[1][i]*Qbv1_7[1]+col1_7[2][i]*Qbv1_7[2]+col1_7[3][i]*Qbv1_7[3]+ col1_7[4][i]*Qbv1_7[4]+ col1_7[5][i]*Qbv1_7[5]+ col1_7[6][i]*Qbv1_7[6]+ col1_7[7][i]*Qbv1_7[7] ) + he2*( col8_11[1][i]*Qbv8_11[1]+ col8_11[2][i]*Qbv8_11[2]+ col8_11[3][i]*Qbv8_11[3]+ col8_11[4][i]*Qbv8_11[4]) + he3*( col12_15[1][i]*Qbv12_15[1]+ col12_15[2][i]*Qbv12_15[2]+ col12_15[3][i]*Qbv12_15[3]+ col12_15[4][i]*Qbv12_15[4]) ;

		y_est[i] = yk1[i] + h*( col1_7[1][i]*bv1_7[1]+col1_7[2][i]*bv1_7[2]+col1_7[3][i]*bv1_7[3]+ col1_7[4][i]*bv1_7[4]+ col1_7[5][i]*bv1_7[5] +col1_7[6][i]*bv1_7[6]+col1_7[7][i]*bv1_7[7] ) + he2*( col8_11[1][i]*bv8_11[1]+ col8_11[2][i]*bv8_11[2]+ col8_11[3][i]*bv8_11[3]+ col8_11[4][i]*bv8_11[4]) + he3*( col12_15[1][i]*bv12_15[1]+ col12_15[2][i]*bv12_15[2]+ col12_15[3][i]*bv12_15[3]+ col12_15[4][i]*bv12_15[4]) ;
		
		if(fabs(ynew[i]-y_est[i]) > err_est)
					err_est=fabs(ynew[i]-y_est[i]);		
	    }
			
hold=h;
if( err_est >  TOL){
	//  (the computed step (hold) is rejected, compute smaller h )
	//       h =  max(  hold *0.5, hmin) ;
				
	stsize =  max(  hold*0.7*stfac *pow(TOL/err_est , 1.0/14.0), hmin) ;			
	nreje=nreje+3 ;	
				
//  **** SECTION: determ. quant. step size h ***********

   hold=preh ;

   if (   stsize < hold *0.15 ) {
        nstsize = 0.1* hold ;
        } 
   if ( (hold *0.15 <= stsize) & ( stsize < hold*0.25 ) ){ 
        nstsize = 0.2* hold ; 
		}
   if ( (hold *0.25 <= stsize) & ( stsize < hold*0.35 ) ){ 
        nstsize = 0.3* hold ; 
		}
   if (  (hold *0.35 <= stsize)& ( stsize < hold*0.45 ) ){ 
        nstsize = 0.4* hold ; 
		}
   if (  (hold *0.45 <= stsize)& ( stsize < hold*0.55 ) ){ 
        nstsize = 0.5* hold ;
		}
   if (  (hold *0.55 <= stsize)& ( stsize < hold*0.65 ) ){ 
        nstsize = 0.6* hold ; 
		}
   if (  (hold *0.65 <= stsize)& ( stsize < hold*0.75 ) ){ 
        nstsize = 0.7* hold ; 
		}
   if (  (hold *0.75 <= stsize)& ( stsize < hold*0.85 ) ){ 
        nstsize = 0.8* hold ; 
		}
   if (  (hold *0.85 <= stsize)& ( stsize < hold* 0.95 ) ){
        nstsize = 0.9* hold ; 
		}
   if (  (hold *0.95 <= stsize)& ( stsize < hold* 1.05 ) ){
        nstsize = 1.0* hold ; 
		} 
   if (  (hold *1.05 <= stsize)& ( stsize < hold* 1.15 ) ){
        nstsize = 1.1* hold ; 
		}
   if (  (hold *1.15  <= stsize)& ( stsize < hold *1.250) ){
        nstsize =  1.2*hold ; 
		}
   if (  (hold *1.25  <= stsize)& ( stsize < hold *1.40) ){
        nstsize =  1.3*hold ; 
		}
   if (  (hold*1.40  <= stsize)& ( stsize < hold *1.60) ){ 
        nstsize =  1.5*hold ; 
		}		
   if (  (hold*1.60  <= stsize)& ( stsize < hold *1.80) ){ 
        nstsize =  1.7*hold ; 
		}
   if   (hold *1.80 <= stsize){  
        nstsize = 1.9*  hold ; 
		}  
   h = nstsize;
   
// (import: j=j-2)   
   j=j-2 ;																
  }
// (end if err_est > TOL) ***********************								
			
}
// (***************** end  LOOP REPEAT UNTIL SUCCESS    while ( err_est > TOL ) ******************		
		tnew = tc + h;
		fpp(tnew, ynew,nbcmp,fnewRes) ;
		
		for(int i=1;i<=nbcmp;i++)
			col1_7[1][i]=fnewRes[0][i] ;
				
		err_est = 0.0 ;

    he2=h*h ;
    he3=he2*h ;

// note: col8_11[5][i]*0 = col8_11[5][i]*b2(b2=0)
	for(int i=1;i<=nbcmp;i++){
		y_est[i] = yk1[i] + h*( col1_7[1][i]*bv1_7[1]+col1_7[2][i]*bv1_7[2]+col1_7[3][i]*bv1_7[3]+ col1_7[4][i]*bv1_7[4]+ col1_7[5][i]*bv1_7[5] +col1_7[6][i]*bv1_7[6] +col1_7[7][i]*bv1_7[7] ) + he2*( col8_11[1][i]*bv8_11[1]+ col8_11[2][i]*bv8_11[2]+ col8_11[3][i]*bv8_11[3]+ col8_11[4][i]*bv8_11[4]) + he3*( col12_15[1][i]*bv12_15[1]+ col12_15[2][i]*bv12_15[2]+ col12_15[3][i]*bv12_15[3]+ col12_15[4][i]*bv12_15[4]) ;
			
		if(fabs(ynew[i]-y_est[i])>err_est)
					err_est=fabs(ynew[i]-y_est[i]);		
	}
		erk=err_est ;				


//         ******* SUBS 2.B STEPSIZE SELECTION: *******
				
		//  (step size formula : h =hold, stsize = hnew)
		if (erk != 0.0){			
			// ACT    (HB../HB11 step cntrl)
			stsize = h * stfac *pow(TOL/erk  , 1.0/14.0  ) ;						
		}else{
			stsize = 1.90*h ;
		} 
		//  (end of step size formula : h =hold, stsize = hnew)
		
		//  constraints that stsize must satisfy :
		//     (a) h( next step) = max (h( next step), hmin, 0.5* h(last step))
		//     (b) h( next step) = min (h( next step), hmax, 4 * h(last step))						
		stsize = max ( stsize, 0.1* h ) ;
		stsize = max ( stsize, hmin ) ;
		stsize = min( stsize, hmax) ;
		stsize = min( stsize, 1.9*h) ;

//  ********(( calc. quant. stzise **********
   if (   stsize < h*0.15  ){ 
        nstsize = 0.1* h ; 
		}
   if (  (h *0.15 <= stsize)& ( stsize < h*0.25 ) ){ 
        nstsize = 0.2* h ; 
		}		
   if (  (h *0.25 <= stsize)& ( stsize < h*0.35 ) ){ 
        nstsize = 0.3* h ; 
		}
   if (  (h *0.35 <= stsize)& ( stsize < h*0.45 ) ){ 
        nstsize = 0.4* h ; 
		}
   if (  (h *0.45 <= stsize)& ( stsize < h*0.55 ) ){ 
        nstsize = 0.5* h ; 
		}
   if (  (h *0.55 <= stsize)& ( stsize < h*0.65 ) ){ 
        nstsize = 0.6* h ; 
		}
   if (  (h *0.65 <= stsize)& ( stsize < h*0.75 ) ){ 
        nstsize = 0.7* h ; 
		}
   if (  (h *0.75 <= stsize)& ( stsize < h*0.85 ) ){ 
        nstsize = 0.8* h ; 
		}
   if (  (h *0.85 <= stsize)& ( stsize < h* 0.95 ) ){ 
        nstsize = 0.9* h ; 
		}
   if (  (h *0.95 <= stsize)& ( stsize < h* 1.05 ) ){ 
        nstsize = 1.0* h ; 
		}
   if (  (h *1.05 <= stsize)& ( stsize < h* 1.15 ) ){ 
        nstsize = 1.1* h ; 
		}
   if (  (h*1.15  <= stsize)& ( stsize < h *1.25 ) ){ 
        nstsize =  1.2*h ; 
		}
   if (  (h*1.25  <= stsize)& ( stsize < h *1.4 ) ){ 
        nstsize =  1.3*h ; 
		}

   if (  (h*1.4  <= stsize)& ( stsize < h *1.60) ){ 
        nstsize =  1.5*h ; 
		}
   if (  (h*1.6  <= stsize)& ( stsize < h *1.80) ){ 
        nstsize =  1.7*h ; 
		}
   if   (h *1.80 <= stsize){  
        nstsize = 1.9*  h ; 
		}
  
  stsize = nstsize;

//  ******** calc. quant. stzise )) **********
		
		tc=tnew;
		jpnbposp1=j+nbpos+1;
			
		for(int i=1;i<=nbcmp;i++)	{
			yc[i]=ynew[i];
			yvals[jpnbposp1][i] =  yc[i];
			fvals[jpnbposp1][i]=  fnewRes[0][i] ;
			f2vals[jpnbposp1][i]=  fnewRes[1][i] ;
			f3vals[jpnbposp1][i]=  fnewRes[2][i] ;
			
		}
		tvals[jpnbposp1] =  tc;					
		hnew=stsize ;
		
		//  (END OF INTEG STEP)

//   ******* SUBS 2.C INTERPOLATION *******
//      if tc > tend ,  recalculate tc, yc using interpolation
//  
//                      recalculate 
//                        (*) j
//                        (*) tvals[jpnbposp1] =tnew
//			              (*) yvals[jpnbposp1][i] =  ynew  ;

 if( tc > tend){

       if (xkm1 > tend)
           j=j-2 ;
        else if (xk1 > tend)
           j=j-1 ;               
		   
       jpnbpos=j+nbpos;
       m14jd=jpnbpos-14  ;  m13jd=jpnbpos-13  ;  m12jd=jpnbpos-12  ;m11jd=jpnbpos-11 ;m10jd=jpnbpos-10  ;m9jd=jpnbpos-9  ;
       m8jd=jpnbpos-8  ;m7jd=jpnbpos-7  ;
       m6jd=jpnbpos-6  ;m5jd=jpnbpos-5  ;m4jd=jpnbpos-4  ;m3jd=jpnbpos-3  ;
       m2jd=jpnbpos-2  ;m1jd=jpnbpos-1  ;m0jd=jpnbpos  ;

       xkm14 =tvals[m14jd ] ;       xkm13 =tvals[m13jd ] ;       xkm12 =tvals[m12jd ] ;       xkm11 =tvals[m11jd ] ;
       xkm10 =tvals[m10jd ] ;       xkm9 = tvals[m9jd ] ;       xkm8 = tvals[m8jd ] ;       xkm7 = tvals[m7jd ] ;
       xkm6 = tvals[m6jd ] ;       xkm5 = tvals[m5jd ] ;       xkm4 = tvals[m4jd ] ;       xkm3 = tvals[m3jd ] ;
       xkm2 = tvals[m2jd ] ;       xkm1 = tvals[m1jd ] ;        xk1 =  tvals[m0jd ] ;       xk2 = tvals[jpnbpos] + c2* h;

       tc=xk1 ;
       xk3 = tvals[jpnbpos] + c3*h;

		for(int i=1;i<=nbcmp;i++)	{
			yk1[i] = yvals[m0jd ][i]  ;
			}
			   
        xtab[1] = tend  ;
        xtab[2] = xkm11 ;
        xtab[3] = xkm10 ;
        xtab[4] = xkm9  ;
        xtab[5] = xkm8  ;
        xtab[6] = xkm7  ;
        xtab[7] = xkm6  ;
        xtab[8] = xkm5  ;
        xtab[9] = xkm4  ;
        xtab[10] =xkm3  ;
        xtab[11] =xkm2  ;
        xtab[12] =xkm1  ;
        xtab[13] =xk1   ;
        xtab[14] =xk1+h   ;

       // calculate y_int at tend using interpolation: call "id123O14"

	   id123O14( xtab, b_v) ;
	   //b_v is the output vector : vector of coeffs of the interpolating polyn. 
	
	for(int i=1;i<=nbcmp;i++){
        // at tend, value of interp polyn. =  ynew[]=y_int[] 
		ynew[i] = yk1[i] + h*( fvals[m0jd+1 ][i] *b_v[1] +fvals[m0jd ][i] *b_v[2] + fvals[m1jd ][i] *b_v[3] +fvals[m2jd ][i] *b_v[4] +fvals[m3jd ][i]*b_v[5] ) + he2*( f2vals[m0jd+1][i] *b_v[6] + f2vals[m0jd ][i]*b_v[7] +		  f2vals[m1jd ][i]*b_v[8] + f2vals[m2jd ][i]*b_v[9] +f2vals[m3jd ][i]*b_v[10] ) + he3*( f3vals[m0jd+1][i] *b_v[11] + f3vals[m0jd ][i]*b_v[12] + f3vals[m1jd ][i]*b_v[13] + f3vals[m2jd ][i]*b_v[14] ) ;
	    }
       tnew=tend ;
       jpnbposp1=j+nbpos+1;
       tvals[jpnbposp1 ] =  tnew;
	   
		for(int i=1;i<=nbcmp;i++)	{
			yvals[jpnbposp1][i] =  ynew[i];
			}
       }
//    (end if( tc > tend))		
		//  (yvals,tvals,.. UPDATE STEP)		
		prevh = h ;
		j++;
	}	
   //  ************** (END  MAIN LOOP TO ADVANCE INTEGRATION FROM x_n to x_{n+3} PER ITERATION) *****				
										
	// print out statistics
	// with index j-1,  yvals(n+1+nbpos, . )  cont solution
	int n=j-1;	
	int NFE = nbinitst*8 +(n+1+nreje-nbinitst)*nbeps;
		
	res[1]=NFE;
	// nb succes step:
	res[2]=n+1 ;
	res[3]=nreje ;	
	//freeing memory
	for(int i=0;i<=2;i++)
		delete fnewRes[i];
	delete fnewRes;
	
}
